package tst;


import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

import junit.framework.TestCase;
import net.sqlind.SQLQueryMapper.GenericQueryHandler;
import net.sqlind.SQLQueryMapper.QueryBehavior;
import net.sqlind.SQLQueryMapper.SQLMapperException;
import net.sqlind.SQLQueryMapper.GenericQueryHandler.BeanWiringBehavior;

/**
 * Parent class for all Mapper projects
 * @author Rives Davy
 */
public abstract class SQLindTester extends TestCase {

	protected static final String TST_QUERIES_XML = "tst/tstQueries.xml";
	boolean running;
	boolean profiling;
	Random rnd;
	DBTestManager db;
	List<Bean> testDeepList;

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		testDeepList = new ArrayList<Bean>();
		running = true;
		profiling = false;
		db = DBTestManager.getInstance();
		rnd = new Random();
	}

	// Tool Classes

	protected class Counter {
		int i = 0;
	}

	protected abstract class Timer {
		double sum = 0.0;
		int hit = 0;

		protected double getAverage() {
			return sum / hit;
		}

		protected void go() throws SQLMapperException {

			Date now = new Date();
			double start = now.getTime();
			profile();
			now = new Date();
			double time = now.getTime() - start;
			if (profiling) {
				sum = sum + time;
				hit++;
			}
		}

		protected abstract void profile() throws SQLMapperException;

	}

	
		protected abstract class TestThread extends Thread implements Runnable {
		boolean ended;

		public void run() {

			final Object cnxObj = getCnxObj();
			try {
				doInit(cnxObj);
				while (running) {
					sleep(doRunning(cnxObj));
				}
				doEnd();
			} catch (Exception e) {
				e.printStackTrace();
				fail("ERROR");
			} finally {
				
				ended = true;

			}
		}

		protected abstract void doInit(Object cnxObj) throws SQLMapperException;

		protected abstract int doRunning(Object cnxObj) throws SQLMapperException;

		protected abstract void doEnd();

	}
		public void testProfiling() {
		final Object cnxObj = getCnxObj();
		try {
			Timer timerReflectWiringPolicy = new Timer() {
				@Override
				public void profile() throws SQLMapperException {
				
					performDeepReflect(cnxObj, new ArrayList<Bean>(), "testLoad1", "Test");
				}
			};
			Timer timerTransfoWiringPolicy = new Timer() {
				@Override
				public void profile() throws SQLMapperException {
					performDeepTransfo(cnxObj, new ArrayList<Bean>(), "testLoad2", "Test");
				}
			};
			// cache
			timerReflectWiringPolicy.go();
			timerTransfoWiringPolicy.go();
			//
			System.out.println("Start to profile wiring policies ...");
			profiling = true;
			for (int i = 0; i < 10000; i++) {
				timerReflectWiringPolicy.go();
				timerTransfoWiringPolicy.go();
			}

			System.out.println(" reflect wiring policy :" + timerReflectWiringPolicy.getAverage() + " ms");
			System.out.println(" transfo wiring policy :" + timerTransfoWiringPolicy.getAverage() + " ms");
			assertTrue(timerReflectWiringPolicy.getAverage() > timerTransfoWiringPolicy.getAverage());
			System.out.println(" transfo gain  :"
					+ Math.round((timerReflectWiringPolicy.getAverage() - timerTransfoWiringPolicy.getAverage())
							/ timerReflectWiringPolicy.getAverage() * 100) + "%");

			System.out.println("terminated.");
		} catch (Exception e) {
			e.printStackTrace();
			fail("ERROR");
			
		} finally {
			try {
				closeCnxObj(cnxObj);
			} catch (Exception e) {
			}

		}
}
		
		protected class TestThreadSafe extends TestThread {

			@Override
			protected void doEnd() {}

			@Override
			protected void doInit(Object cnxObj) throws SQLMapperException {}

			@Override
			protected int doRunning(Object cnxObj) throws SQLMapperException {
				GenericQueryHandler q = getSQLQueryHandlerImpl(null, "testThread");
					//SQLQueryMapper.getInstance().getSQLQuery(TST_QUERIES_XML, null, "testThread");
				final String[] expected = new String[] { null, null, "C", "R", "Z" };
				for (int i = 1; i < 6; i++) {
					final int index = i-1;
					q.setParameter("key", i);
				
					try {sleep(15);} catch (InterruptedException e) {e.printStackTrace();}
					BeanWiringBehavior<Bean> wirer = q.new BeanWiringBehavior<Bean>(Bean.class) {
						int recordNum = 0;

						@Override
						public void doForEachRupture(Bean bean) {
							assertEquals("Bad value", expected[index], bean.getAggrbean().getLabel());
						}

						@Override
						public void doForEachSubRupture(Object bean) {
						}

						@Override
						public void doForEachRecord(Bean bean) {
							recordNum++;
							assertTrue("Only one record should have been retrieved.", recordNum ==1);
						}
					};
					
					performQuery(cnxObj,q,wirer);
					//q.performSelectQuery(con,wirer );
				}
			
				return rnd.nextInt(15);

			}
		}

		
		public void testThreadSafe() throws SQLMapperException {
			try {
				System.out.println("Start testing thread safe behavior ...");
				int threadNum = 15;
				List<TestThreadSafe> lt = new ArrayList<TestThreadSafe>();
				for (int i = 0; i < threadNum; i++) {
					TestThreadSafe t = new TestThreadSafe();
					lt.add(t);
					t.start();
				}

				Thread.sleep(50000);
				running = false;
				while(lt.size()>0){
					if(lt.get(0).ended)
						lt.remove(0);
					Thread.sleep(200);
				}
				System.out.println("terminated");
			} catch (Exception e) {
				fail("ERROR");
				e.printStackTrace();
			}finally{
				try {
					db.getCnx().close();
				} catch (SQLException e) {
					
					e.printStackTrace();
				}
			}
		}
		
		protected Counter performDeepTransfo(Object cnxObj, final List<Bean> results, String queryId, String injectID)
		throws SQLMapperException {
	final Counter subRuptureCount = new Counter();

	//SQLQueryHandler q = SQLQueryMapper.getInstance().getSQLQuery(TST_QUERIES_XML, null, queryId);
	GenericQueryHandler q = getSQLQueryHandlerImpl(null, queryId);

	q.setInjection("table", injectID);
	performQuery(cnxObj,q, q.new BeanWiringBehavior<Bean>(Bean.class) {
		@Override
		public void doForEachRupture(Bean bean) {
			results.add(bean);
		}

		@Override
		public void doForEachSubRupture(Object bean) {
			assertNotNull("SubRupture is abnormal", bean);
			subRuptureCount.i++;
		}

		@Override
		public void doForEachRecord(Bean bean) {
		}
	});
	return subRuptureCount;
}

		
		protected Counter performDeepReflect(Object cnxObj, final List<Bean> results, String queryId, String injectID)
		throws SQLMapperException {
	
	final Counter subRuptureCount = new Counter();
	GenericQueryHandler q = getSQLQueryHandlerImpl(null, queryId);
	//SQLQueryHandler q = SQLQueryMapper.getInstance().getSQLQuery(TST_QUERIES_XML, null, queryId);
	q.setInjection("table", injectID);
	performQuery(cnxObj, q, q.new BeanWiringBehavior<Bean>(Bean.class) {
		@Override
		public void doForEachRupture(Bean bean) {
			results.add(bean);
		}

		@Override
		public void doForEachSubRupture(Object bean) {
			assertNotNull("SubRupture is abnormal", bean);
			subRuptureCount.i++;
		}

		@Override
		public void doForEachRecord(Bean bean) {
		}
	});
	return subRuptureCount;
}
		public void testDeepTransfoWiringPolicy() throws Exception {
			Object cnxObject = null;
			try {
			
				cnxObject = getCnxObj();
				final List<Bean> results = new ArrayList<Bean>();
				final Counter subRuptureCount = performDeepTransfo(cnxObject, results, "test2",getTestInject());

				assertEquals("Number of subruptures is false", 10, subRuptureCount.i);
				assertEquals("Number of beans doesn't match", 5, results.size());
				assertEquals("Value of sub ruptures doesn't match", "v3", results.get(3).getBean().get(2).getLabel());
				for (int i = 0; i < testDeepList.size(); i++)
					assertEquals("Differences between reflect and transfo policy", testDeepList.get(i).toString(), results
							.get(i).toString());
			} catch (Exception e) {
				e.printStackTrace();
				fail("ERROR");
				if (cnxObject != null)
					closeCnxObj(cnxObject);
			}
		}
		public void testDeepReflectWiringPolicy() throws Exception {
			Object cnxObject = null;
			try {
			
				cnxObject = getCnxObj();
				final List<Bean> results = new ArrayList<Bean>();
				final Counter subRuptureCount = performDeepReflect(cnxObject, results, "test1",getTestInject());
				assertEquals("Number of subruptures is false", 10, subRuptureCount.i);
				assertEquals("Number of main ruptures doesn't match", 5, results.size());
				assertEquals("Number of sub ruptures doesn't match", 3, results.get(3).getBean().size());
				testDeepList.addAll(results);
			} catch (Exception e) {
				e.printStackTrace();
				fail("ERROR");
				if (cnxObject != null)
					closeCnxObj(cnxObject);
			}
		}
		
		protected abstract String getTestInject();
		protected abstract Object getCnxObj();
		
		protected abstract void closeCnxObj(Object cnxObj) throws SQLException;
		protected abstract GenericQueryHandler getSQLQueryHandlerImpl(String schema,String queryId ,String...  params) throws SQLMapperException;
		protected abstract void performQuery(Object cnxObj,GenericQueryHandler handler,QueryBehavior behavior) throws SQLMapperException;

	
}
