/**
 * <h1>Amazon Lambda Node.js Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This library provides constructs for Node.js Lambda functions.
 * <p>
 * <h2>Node.js Function</h2>
 * <p>
 * The <code>NodejsFunction</code> construct creates a Lambda function with automatic transpiling and bundling
 * of TypeScript or Javascript code. This results in smaller Lambda packages that contain only the
 * code and dependencies needed to run the function.
 * <p>
 * It uses <a href="https://esbuild.github.io/">esbuild</a> under the hood.
 * <p>
 * <h2>Reference project architecture</h2>
 * <p>
 * The <code>NodejsFunction</code> allows you to define your CDK and runtime dependencies in a single
 * package.json and to collocate your runtime code with your infrastructure code:
 * <p>
 * <blockquote><pre>
 * .
 * ├── lib
 * │   ├── my-construct.api.ts # Lambda handler for API
 * │   ├── my-construct.auth.ts # Lambda handler for Auth
 * │   └── my-construct.ts # CDK construct with two Lambda functions
 * ├── package-lock.json # single lock file
 * ├── package.json # CDK and runtime dependencies defined in a single package.json
 * └── tsconfig.json
 * </pre></blockquote>
 * <p>
 * By default, the construct will use the name of the defining file and the construct's
 * id to look up the entry file. In <code>my-construct.ts</code> above we have:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * // automatic entry look up
 * Object apiHandler = new NodejsFunction(this, "api");
 * Object authHandler = new NodejsFunction(this, "auth");
 * </pre></blockquote>
 * <p>
 * Alternatively, an entry file and handler can be specified:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "MyFunction")
 *         .entry("/path/to/my/file.ts")// accepts .js, .jsx, .ts and .tsx files
 *         .handler("myExportedFunc")
 *         .build();
 * </pre></blockquote>
 * <p>
 * For monorepos, the reference architecture becomes:
 * <p>
 * <blockquote><pre>
 * .
 * ├── packages
 * │   ├── cool-package
 * │   │   ├── lib
 * │   │   │   ├── cool-construct.api.ts
 * │   │   │   ├── cool-construct.auth.ts
 * │   │   │   └── cool-construct.ts
 * │   │   ├── package.json # CDK and runtime dependencies for cool-package
 * │   │   └── tsconfig.json
 * │   └── super-package
 * │       ├── lib
 * │       │   ├── super-construct.handler.ts
 * │       │   └── super-construct.ts
 * │       ├── package.json # CDK and runtime dependencies for super-package
 * │       └── tsconfig.json
 * ├── package-lock.json # single lock file
 * ├── package.json # root dependencies
 * └── tsconfig.json
 * </pre></blockquote>
 * <p>
 * <h2>Customizing the underlying Lambda function</h2>
 * <p>
 * All properties of <code>lambda.Function</code> can be used to customize the underlying <code>lambda.Function</code>.
 * <p>
 * See also the <a href="https://github.com/aws/aws-cdk/tree/master/packages/%40aws-cdk/aws-lambda">AWS Lambda construct library</a>.
 * <p>
 * The <code>NodejsFunction</code> construct automatically <a href="https://docs.aws.amazon.com/sdk-for-javascript/v2/developer-guide/node-reusing-connections.html">reuses existing connections</a>
 * when working with the AWS SDK for JavaScript. Set the <code>awsSdkConnectionReuse</code> prop to <code>false</code> to disable it.
 * <p>
 * <h2>Lock file</h2>
 * <p>
 * The <code>NodejsFunction</code> requires a dependencies lock file (<code>yarn.lock</code>, <code>pnpm-lock.yaml</code> or
 * <code>package-lock.json</code>). When bundling in a Docker container, the path containing this lock file is
 * used as the source (<code>/asset-input</code>) for the volume mounted in the container.
 * <p>
 * By default, the construct will try to automatically determine your project lock file.
 * Alternatively, you can specify the <code>depsLockFilePath</code> prop manually. In this
 * case you need to ensure that this path includes <code>entry</code> and any module/dependencies
 * used by your function. Otherwise bundling will fail.
 * <p>
 * <h2>Local bundling</h2>
 * <p>
 * If <code>esbuild</code> is available it will be used to bundle your code in your environment. Otherwise,
 * bundling will happen in a <a href="https://gallery.ecr.aws/sam/build-nodejs12.x">Lambda compatible Docker container</a>.
 * <p>
 * For macOS the recommendend approach is to install <code>esbuild</code> as Docker volume performance is really poor.
 * <p>
 * <code>esbuild</code> can be installed with:
 * <p>
 * <blockquote><pre>
 * $ npm install --save-dev esbuild&#64;0
 * </pre></blockquote>
 * <p>
 * OR
 * <p>
 * <blockquote><pre>
 * $ yarn add --dev esbuild&#64;0
 * </pre></blockquote>
 * <p>
 * To force bundling in a Docker container even if <code>esbuild</code> is available in your environment,
 * set <code>bundling.forceDockerBundling</code> to <code>true</code>. This is useful if your function relies on node
 * modules that should be installed (<code>nodeModules</code> prop, see <a href="#install-modules">below</a>) in a Lambda
 * compatible environment. This is usually the case with modules using native dependencies.
 * <p>
 * <h2>Working with modules</h2>
 * <p>
 * <h3>Externals</h3>
 * <p>
 * By default, all node modules are bundled except for <code>aws-sdk</code>. This can be configured by specifying
 * <code>bundling.externalModules</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "externalModules", asList("aws-sdk", "cool-module")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Install modules</h3>
 * <p>
 * By default, all node modules referenced in your Lambda code will be bundled by <code>esbuild</code>.
 * Use the <code>nodeModules</code> prop under <code>bundling</code> to specify a list of modules that should not be
 * bundled but instead included in the <code>node_modules</code> folder of the Lambda package. This is useful
 * when working with native dependencies or when <code>esbuild</code> fails to bundle a module.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "nodeModules", asList("native-module", "other-module")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The modules listed in <code>nodeModules</code> must be present in the <code>package.json</code>'s dependencies or
 * installed. The same version will be used for installation. The lock file (<code>yarn.lock</code>,
 * <code>pnpm-lock.yaml</code> or <code>package-lock.json</code>) will be used along with the right installer (<code>yarn</code>,
 * <code>pnpm</code> or <code>npm</code>).
 * <p>
 * When working with <code>nodeModules</code> using native dependencies, you might want to force bundling in a
 * Docker container even if <code>esbuild</code> is available in your environment. This can be done by setting
 * <code>bundling.forceDockerBundling</code> to <code>true</code>.
 * <p>
 * <h2>Configuring <code>esbuild</code></h2>
 * <p>
 * The <code>NodejsFunction</code> construct exposes some <a href="https://esbuild.github.io/api/#build-api">esbuild options</a>
 * via properties under <code>bundling</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "minify", true, // minify code, defaults to false
 *                 "sourceMap", true, // include source map, defaults to false
 *                 "sourceMapMode", SourceMapMode.getINLINE(), // defaults to SourceMapMode.DEFAULT
 *                 "target", "es2020", // target environment for the generated JavaScript code
 *                 "loader", Map.of(// Use the 'dataurl' loader for '.png' files
 *                         ".png", "dataurl"),
 *                 "define", Map.of(// Replace strings during build time
 *                         "process.env.API_KEY", JSON.stringify("xxx-xxxx-xxx"),
 *                         "process.env.PRODUCTION", JSON.stringify(true),
 *                         "process.env.NUMBER", JSON.stringify(123)),
 *                 "logLevel", LogLevel.getSILENT(), // defaults to LogLevel.WARNING
 *                 "keepNames", true, // defaults to false
 *                 "tsconfig", "custom-tsconfig.json", // use custom-tsconfig.json instead of default,
 *                 "metafile", true, // include meta file, defaults to false
 *                 "banner", "/* comments *{@literal /}", // requires esbuild &gt;= 0.9.0, defaults to none
 *                 "footer", "/* comments *{@literal /}"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Command hooks</h2>
 * <p>
 * It is possible to run additional commands by specifying the <code>commandHooks</code> prop:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler-with-commands")
 *         .bundling(Map.of(
 *                 "commandHooks", Map.of(
 *                         // Copy a file so that it will be included in the bundled asset
 *                         public Array afterBundling(String inputDir, String outputDir) {return asList(String.format("cp %s/my-binary.node %s", inputDir, outputDir))
 *                         })))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following hooks are available:
 * <p>
 * <ul>
 * <li><code>beforeBundling</code>: runs before all bundling commands</li>
 * <li><code>beforeInstall</code>: runs before node modules installation</li>
 * <li><code>afterBundling</code>: runs after all bundling commands</li>
 * </ul>
 * <p>
 * They all receive the directory containing the lock file (<code>inputDir</code>) and the
 * directory where the bundled asset will be output (<code>outputDir</code>). They must return
 * an array of commands to run. Commands are chained with <code>&amp;&amp;</code>.
 * <p>
 * The commands will run in the environment in which bundling occurs: inside the
 * container for Docker bundling or on the host OS for local bundling.
 * <p>
 * <h2>Customizing Docker bundling</h2>
 * <p>
 * Use <code>bundling.environment</code> to define environments variables when <code>esbuild</code> runs:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "environment", Map.of(
 *                         "NODE_ENV", "production")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>bundling.buildArgs</code> to pass build arguments when building the Docker bundling image:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "buildArgs", Map.of(
 *                         "HTTPS_PROXY", "https://127.0.0.1:3001")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>bundling.dockerImage</code> to use a custom Docker bundling image:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(Map.of(
 *                 "dockerImage", cdk.DockerImage.fromBuild("/path/to/Dockerfile")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This image should have <code>esbuild</code> installed <strong>globally</strong>. If you plan to use <code>nodeModules</code> it
 * should also have <code>npm</code>, <code>yarn</code> or <code>pnpm</code> depending on the lock file you're using.
 * <p>
 * Use the <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-lambda-nodejs/lib/Dockerfile">default image provided by <code>&#64;aws-cdk/aws-lambda-nodejs</code></a>
 * as a source of inspiration.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.lambda.nodejs;
