/**
 * <h1>Amazon Lambda Python Library</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Experimental" src="https://img.shields.io/badge/cdk--constructs-experimental-important.svg?style=for-the-badge">
 * <p>
 * <blockquote>
 * <p>
 * The APIs of higher level constructs in this module are experimental and under active development.
 * They are subject to non-backward compatible changes or removal in any future version. These are
 * not subject to the <a href="https://semver.org/">Semantic Versioning</a> model and breaking changes will be
 * announced in the release notes. This means that while you may use them, you may need to update
 * your source code when upgrading to a newer version of this package.
 * <p>
 * </blockquote>
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * This library provides constructs for Python Lambda functions.
 * <p>
 * To use this module, you will need to have Docker installed.
 * <p>
 * <h2>Python Function</h2>
 * <p>
 * Define a <code>PythonFunction</code>:
 * <p>
 * <blockquote><pre>
 * PythonFunction.Builder.create(this, "MyFunction")
 *         .entry("/path/to/my/function") // required
 *         .runtime(Runtime.PYTHON_3_8) // required
 *         .index("my_index.py") // optional, defaults to 'index.py'
 *         .handler("my_exported_func")
 *         .build();
 * </pre></blockquote>
 * <p>
 * All other properties of <code>lambda.Function</code> are supported, see also the <a href="https://github.com/aws/aws-cdk/tree/main/packages/%40aws-cdk/aws-lambda">AWS Lambda construct library</a>.
 * <p>
 * <h2>Python Layer</h2>
 * <p>
 * You may create a python-based lambda layer with <code>PythonLayerVersion</code>. If <code>PythonLayerVersion</code> detects a <code>requirements.txt</code>
 * or <code>Pipfile</code> or <code>poetry.lock</code> with the associated <code>pyproject.toml</code> at the entry path, then <code>PythonLayerVersion</code> will include the dependencies inline with your code in the
 * layer.
 * <p>
 * Define a <code>PythonLayerVersion</code>:
 * <p>
 * <blockquote><pre>
 * PythonLayerVersion.Builder.create(this, "MyLayer")
 *         .entry("/path/to/my/layer")
 *         .build();
 * </pre></blockquote>
 * <p>
 * A layer can also be used as a part of a <code>PythonFunction</code>:
 * <p>
 * <blockquote><pre>
 * PythonFunction.Builder.create(this, "MyFunction")
 *         .entry("/path/to/my/function")
 *         .runtime(Runtime.PYTHON_3_8)
 *         .layers(List.of(
 *             PythonLayerVersion.Builder.create(this, "MyLayer")
 *                     .entry("/path/to/my/layer")
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Packaging</h2>
 * <p>
 * If <code>requirements.txt</code>, <code>Pipfile</code> or <code>poetry.lock</code> exists at the entry path, the construct will handle installing all required modules in a <a href="https://gallery.ecr.aws/sam/build-python3.7">Lambda compatible Docker container</a> according to the <code>runtime</code> and with the Docker platform based on the target architecture of the Lambda function.
 * <p>
 * Python bundles are only recreated and published when a file in a source directory has changed.
 * Therefore (and as a general best-practice), it is highly recommended to commit a lockfile with a
 * list of all transitive dependencies and their exact versions. This will ensure that when any dependency version is updated, the bundle asset is recreated and uploaded.
 * <p>
 * To that end, we recommend using [<code>pipenv</code>] or [<code>poetry</code>] which have lockfile support.
 * <p>
 * <ul>
 * <li><a href="https://pipenv-fork.readthedocs.io/en/latest/basics.html#example-pipfile-lock"><code>pipenv</code></a></li>
 * <li><a href="https://python-poetry.org/docs/basic-usage/#commit-your-poetrylock-file-to-version-control"><code>poetry</code></a></li>
 * </ul>
 * <p>
 * Packaging is executed using the <code>Packaging</code> class, which:
 * <p>
 * <ol>
 * <li>Infers the packaging type based on the files present.</li>
 * <li>If it sees a <code>Pipfile</code> or a <code>poetry.lock</code> file, it exports it to a compatible <code>requirements.txt</code> file with credentials (if they're available in the source files or in the bundling container).</li>
 * <li>Installs dependencies using <code>pip</code>.</li>
 * <li>Copies the dependencies into an asset that is bundled for the Lambda package.</li>
 * </ol>
 * <p>
 * <strong>Lambda with a requirements.txt</strong>
 * <p>
 * <blockquote><pre>
 * .
 * ├── lambda_function.py # exports a function named 'handler'
 * ├── requirements.txt # has to be present at the entry path
 * </pre></blockquote>
 * <p>
 * <strong>Lambda with a Pipfile</strong>
 * <p>
 * <blockquote><pre>
 * .
 * ├── lambda_function.py # exports a function named 'handler'
 * ├── Pipfile # has to be present at the entry path
 * ├── Pipfile.lock # your lock file
 * </pre></blockquote>
 * <p>
 * <strong>Lambda with a poetry.lock</strong>
 * <p>
 * <blockquote><pre>
 * .
 * ├── lambda_function.py # exports a function named 'handler'
 * ├── pyproject.toml # your poetry project definition
 * ├── poetry.lock # your poetry lock file has to be present at the entry path
 * </pre></blockquote>
 * <p>
 * <h2>Custom Bundling</h2>
 * <p>
 * Custom bundling can be performed by passing in additional build arguments that point to index URLs to private repos, or by using an entirely custom Docker images for bundling dependencies. The build args currently supported are:
 * <p>
 * <ul>
 * <li><code>PIP_INDEX_URL</code></li>
 * <li><code>PIP_EXTRA_INDEX_URL</code></li>
 * <li><code>HTTPS_PROXY</code></li>
 * </ul>
 * <p>
 * Additional build args for bundling that refer to PyPI indexes can be specified as:
 * <p>
 * <blockquote><pre>
 * String entry = "/path/to/function";
 * DockerImage image = DockerImage.fromBuild(entry);
 * 
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder()
 *                 .buildArgs(Map.of("PIP_INDEX_URL", "https://your.index.url/simple/", "PIP_EXTRA_INDEX_URL", "https://your.extra-index.url/simple/"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If using a custom Docker image for bundling, the dependencies are installed with <code>pip</code>, <code>pipenv</code> or <code>poetry</code> by using the <code>Packaging</code> class. A different bundling Docker image that is in the same directory as the function can be specified as:
 * <p>
 * <blockquote><pre>
 * String entry = "/path/to/function";
 * DockerImage image = DockerImage.fromBuild(entry);
 * 
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder().image(image).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can set additional Docker options to configure the build environment:
 * <p>
 * <blockquote><pre>
 * String entry = "/path/to/function";
 * 
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder()
 *                 .network("host")
 *                 .securityOpt("no-new-privileges")
 *                 .user("user:group")
 *                 .volumesFrom(List.of("777f7dc92da7"))
 *                 .volumes(List.of(DockerVolume.builder().hostPath("/host-path").containerPath("/container-path").build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Custom Bundling with Code Artifact</h2>
 * <p>
 * To use a Code Artifact PyPI repo, the <code>PIP_INDEX_URL</code> for bundling the function can be customized (requires AWS CLI in the build environment):
 * <p>
 * <blockquote><pre>
 * import child.process.execSync;
 * 
 * 
 * String entry = "/path/to/function";
 * DockerImage image = DockerImage.fromBuild(entry);
 * 
 * String domain = "my-domain";
 * String domainOwner = "111122223333";
 * String repoName = "my_repo";
 * String region = "us-east-1";
 * String codeArtifactAuthToken = execSync(String.format("aws codeartifact get-authorization-token --domain %s --domain-owner %s --query authorizationToken --output text", domain, domainOwner)).toString().trim();
 * 
 * String indexUrl = String.format("https://aws:%s&#64;%s-%s.d.codeartifact.%s.amazonaws.com/pypi/%s/simple/", codeArtifactAuthToken, domain, domainOwner, region, repoName);
 * 
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder()
 *                 .environment(Map.of("PIP_INDEX_URL", indexUrl))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The index URL or the token are only used during bundling and thus not included in the final asset. Setting only environment variable for <code>PIP_INDEX_URL</code> or <code>PIP_EXTRA_INDEX_URL</code> should work for accesing private Python repositories with <code>pip</code>, <code>pipenv</code> and <code>poetry</code> based dependencies.
 * <p>
 * If you also want to use the Code Artifact repo for building the base Docker image for bundling, use <code>buildArgs</code>. However, note that setting custom build args for bundling will force the base bundling image to be rebuilt every time (i.e. skip the Docker cache). Build args can be customized as:
 * <p>
 * <blockquote><pre>
 * import child.process.execSync;
 * 
 * 
 * String entry = "/path/to/function";
 * DockerImage image = DockerImage.fromBuild(entry);
 * 
 * String domain = "my-domain";
 * String domainOwner = "111122223333";
 * String repoName = "my_repo";
 * String region = "us-east-1";
 * String codeArtifactAuthToken = execSync(String.format("aws codeartifact get-authorization-token --domain %s --domain-owner %s --query authorizationToken --output text", domain, domainOwner)).toString().trim();
 * 
 * String indexUrl = String.format("https://aws:%s&#64;%s-%s.d.codeartifact.%s.amazonaws.com/pypi/%s/simple/", codeArtifactAuthToken, domain, domainOwner, region, repoName);
 * 
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder()
 *                 .buildArgs(Map.of("PIP_INDEX_URL", indexUrl))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Command hooks</h2>
 * <p>
 * It is  possible to run additional commands by specifying the <code>commandHooks</code> prop:
 * <p>
 * <blockquote><pre>
 * String entry = "/path/to/function";
 * PythonFunction.Builder.create(this, "function")
 *         .entry(entry)
 *         .runtime(Runtime.PYTHON_3_8)
 *         .bundling(BundlingOptions.builder()
 *                 .commandHooks(Map.of(
 *                         // run tests
 *                         public String[] beforeBundling(String inputDir) {
 *                             return List.of("pytest");
 *                         },
 *                         public String[] afterBundling(String inputDir) {
 *                             return List.of("pylint");
 *                         }))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following hooks are available:
 * <p>
 * <ul>
 * <li><code>beforeBundling</code>: runs before all bundling commands</li>
 * <li><code>afterBundling</code>: runs after all bundling commands</li>
 * </ul>
 * <p>
 * They all receive the directory containing the dependencies file (<code>inputDir</code>) and the
 * directory where the bundled asset will be output (<code>outputDir</code>). They must return
 * an array of commands to run. Commands are chained with <code>&amp;&amp;</code>.
 * <p>
 * The commands will run in the environment in which bundling occurs: inside the
 * container for Docker bundling or on the host OS for local bundling.
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.lambda.python.alpha;
