/**
 * <h2>AWS Lambda Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Stable-success.svg?style=for-the-badge" alt="Stability: Stable" /></p>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>This construct library allows you to define AWS Lambda Functions.</p>
 * <pre><code class="language-ts">// This example is in TypeScript, examples in Java are coming soon.
 * import lambda = require('@aws-cdk/aws-lambda');
 * import path = require('path');
 * 
 * const fn = new lambda.Function(this, 'MyFunction', {
 *   runtime: lambda.Runtime.NODEJS_10_X,
 *   handler: 'index.handler',
 *   code: lambda.Code.fromAsset(path.join(__dirname, 'lambda-handler')),
 * });
 * </code></pre>
 * <h3>Handler Code</h3>
 * <p>The <code>lambda.Code</code> class includes static convenience methods for various types of
 * runtime code.</p>
 * <ul>
 * <li><code>lambda.Code.fromBucket(bucket, key[, objectVersion])</code> - specify an S3 object
 * that contains the archive of your runtime code.</li>
 * <li><code>lambda.Code.fromInline(code)</code> - inline the handle code as a string. This is
 * limited to supported runtimes and the code cannot exceed 4KiB.</li>
 * <li><code>lambda.Code.fromAsset(path)</code> - specify a directory or a .zip file in the local
 * filesystem which will be zipped and uploaded to S3 before deployment.</li>
 * </ul>
 * <p>The following example shows how to define a Python function and deploy the code
 * from the local directory <code>my-lambda-handler</code> to it:</p>
 * <pre><code class="language-ts">new lambda.Function(this, 'MyLambda', {
 *   code: lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler')),
 *   handler: 'index.main',
 *   runtime: lambda.Runtime.PYTHON_3_6
 * });
 * </code></pre>
 * <p>When deploying a stack that contains this code, the directory will be zip
 * archived and then uploaded to an S3 bucket, then the exact location of the S3
 * objects will be passed when the stack is deployed.</p>
 * <p>During synthesis, the CDK expects to find a directory on disk at the asset
 * directory specified. Note that we are referencing the asset directory relatively
 * to our CDK project directory. This is especially important when we want to share
 * this construct through a library. Different programming languages will have
 * different techniques for bundling resources into libraries.</p>
 * <h3>Layers</h3>
 * <p>The <code>lambda.LayerVersion</code> class can be used to define Lambda layers and manage
 * granting permissions to other AWS accounts or organizations.</p>
 * <pre><code class="language-ts">const layer = new lambda.LayerVersion(stack, 'MyLayer', {
 *   code: lambda.Code.fromAsset(path.join(__dirname, 'layer-code')),
 *   compatibleRuntimes: [lambda.Runtime.NODEJS_10_X],
 *   license: 'Apache-2.0',
 *   description: 'A layer to test the L2 construct',
 * });
 * 
 * // To grant usage by other AWS accounts
 * layer.addPermission('remote-account-grant', { accountId: awsAccountId });
 * 
 * // To grant usage to all accounts in some AWS Ogranization
 * // layer.grantUsage({ accountId: '*', organizationId });
 * 
 * new lambda.Function(stack, 'MyLayeredLambda', {
 *   code: new lambda.InlineCode('foo'),
 *   handler: 'index.handler',
 *   runtime: lambda.Runtime.NODEJS_10_X,
 *   layers: [layer],
 * });
 * </code></pre>
 * <h2>Event Rule Target</h2>
 * <p>You can use an AWS Lambda function as a target for an Amazon CloudWatch event
 * rule:</p>
 * <pre><code class="language-ts">import targets = require('@aws-cdk/aws-events-targets');
 * rule.addTarget(new targets.LambdaFunction(myFunction));
 * </code></pre>
 * <h3>Event Sources</h3>
 * <p>AWS Lambda supports a <a href="https://docs.aws.amazon.com/lambda/latest/dg/invoking-lambda-function.html">variety of event sources</a>.</p>
 * <p>In most cases, it is possible to trigger a function as a result of an event by
 * using one of the <code>add&lt;Event&gt;Notification</code> methods on the source construct. For
 * example, the <code>s3.Bucket</code> construct has an <code>onEvent</code> method which can be used to
 * trigger a Lambda when an event, such as PutObject occurs on an S3 bucket.</p>
 * <p>An alternative way to add event sources to a function is to use <code>function.addEventSource(source)</code>.
 * This method accepts an <code>IEventSource</code> object. The module <strong>@aws-cdk/aws-lambda-event-sources</strong>
 * includes classes for the various event sources supported by AWS Lambda.</p>
 * <p>For example, the following code adds an SQS queue as an event source for a function:</p>
 * <pre><code class="language-ts">import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 * fn.addEventSource(new SqsEventSource(queue));
 * </code></pre>
 * <p>The following code adds an S3 bucket notification as an event source:</p>
 * <pre><code class="language-ts">import { S3EventSource } from '@aws-cdk/aws-lambda-event-sources';
 * fn.addEventSource(new S3EventSource(bucket, {
 *   events: [ s3.EventType.OBJECT_CREATED, s3.EventType.OBJECT_DELETED ],
 *   filters: [ { prefix: 'subdir/' } ] // optional
 * }));
 * </code></pre>
 * <p>See the documentation for the <strong>@aws-cdk/aws-lambda-event-sources</strong> module for more details.</p>
 * <h3>Lambda with DLQ</h3>
 * <p>A dead-letter queue can be automatically created for a Lambda function by
 * setting the <code>deadLetterQueueEnabled: true</code> configuration.</p>
 * <pre><code class="language-ts">import lambda = require('@aws-cdk/aws-lambda');
 * 
 * const fn = new lambda.Function(this, 'MyFunction', {
 *     runtime: lambda.Runtime.NODEJS_8_10,
 *     handler: 'index.handler',
 *     code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, &quot;hi&quot;); }'),
 *     deadLetterQueueEnabled: true
 * });
 * </code></pre>
 * <p>It is also possible to provide a dead-letter queue instead of getting a new queue created:</p>
 * <pre><code class="language-ts">import lambda = require('@aws-cdk/aws-lambda');
 * import sqs = require('@aws-cdk/aws-sqs');
 * 
 * const dlq = new sqs.Queue(this, 'DLQ');
 * const fn = new lambda.Function(this, 'MyFunction', {
 *     runtime: lambda.Runtime.NODEJS_8_10,
 *     handler: 'index.handler',
 *     code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, &quot;hi&quot;); }'),
 *     deadLetterQueue: dlq
 * });
 * </code></pre>
 * <p>See <a href="https://docs.aws.amazon.com/lambda/latest/dg/dlq.html">the AWS documentation</a>
 * to learn more about AWS Lambdas and DLQs.</p>
 * <h3>Lambda with X-Ray Tracing</h3>
 * <pre><code class="language-ts">import lambda = require('@aws-cdk/aws-lambda');
 * 
 * const fn = new lambda.Function(this, 'MyFunction', {
 *     runtime: lambda.Runtime.NODEJS_8_10,
 *     handler: 'index.handler',
 *     code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, &quot;hi&quot;); }'),
 *     tracing: lambda.Tracing.ACTIVE
 * });
 * </code></pre>
 * <p>See <a href="https://docs.aws.amazon.com/lambda/latest/dg/lambda-x-ray.html">the AWS documentation</a>
 * to learn more about AWS Lambda's X-Ray support.</p>
 * <h3>Lambda with Reserved Concurrent Executions</h3>
 * <pre><code class="language-ts">import lambda = require('@aws-cdk/aws-lambda');
 * 
 * const fn = new lambda.Function(this, 'MyFunction', {
 *     runtime: lambda.Runtime.NODEJS_8_10,
 *     handler: 'index.handler',
 *     code: lambda.Code.fromInline('exports.handler = function(event, ctx, cb) { return cb(null, &quot;hi&quot;); }'),
 *     reservedConcurrentExecutions: 100
 * });
 * </code></pre>
 * <p>See <a href="https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html">the AWS documentation</a>
 * managing concurrency.</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.lambda;
