/**
 * <h1>Tasks for AWS Step Functions</h1>
 * <p>
 * <!--BEGIN STABILITY BANNER-->---
 * <p>
 * <img alt="cdk-constructs: Stable" src="https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge">
 * <p>
 * <hr>
 * <p>
 * <!--END STABILITY BANNER-->
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html">AWS Step Functions</a> is a web service that enables you to coordinate the
 * components of distributed applications and microservices using visual workflows.
 * You build applications from individual components that each perform a discrete
 * function, or task, allowing you to scale and change applications quickly.
 * <p>
 * A <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-task-state.html">Task</a> state represents a single unit of work performed by a state machine.
 * All work in your state machine is performed by tasks.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Table Of Contents</h2>
 * <p>
 * <ul>
 * <li><a href="#tasks-for-aws-step-functions">Tasks for AWS Step Functions</a>
 * <p>
 * <ul>
 * <li><a href="#table-of-contents">Table Of Contents</a></li>
 * <li><a href="#task">Task</a></li>
 * <li><a href="#paths">Paths</a>
 * <p>
 * <ul>
 * <li><a href="#inputpath">InputPath</a></li>
 * <li><a href="#outputpath">OutputPath</a></li>
 * <li><a href="#resultpath">ResultPath</a></li>
 * </ul></li>
 * <li><a href="#task-parameters-from-the-state-json">Task parameters from the state JSON</a></li>
 * <li><a href="#evaluate-expression">Evaluate Expression</a></li>
 * <li><a href="#api-gateway">API Gateway</a>
 * <p>
 * <ul>
 * <li><a href="#call-rest-api-endpoint">Call REST API Endpoint</a></li>
 * <li><a href="#call-http-api-endpoint">Call HTTP API Endpoint</a></li>
 * </ul></li>
 * <li><a href="#athena">Athena</a>
 * <p>
 * <ul>
 * <li><a href="#startqueryexecution">StartQueryExecution</a></li>
 * <li><a href="#getqueryexecution">GetQueryExecution</a></li>
 * <li><a href="#getqueryresults">GetQueryResults</a></li>
 * <li><a href="#stopqueryexecution">StopQueryExecution</a></li>
 * </ul></li>
 * <li><a href="#batch">Batch</a>
 * <p>
 * <ul>
 * <li><a href="#submitjob">SubmitJob</a></li>
 * </ul></li>
 * <li><a href="#codebuild">CodeBuild</a>
 * <p>
 * <ul>
 * <li><a href="#startbuild">StartBuild</a></li>
 * </ul></li>
 * <li><a href="#dynamodb">DynamoDB</a>
 * <p>
 * <ul>
 * <li><a href="#getitem">GetItem</a></li>
 * <li><a href="#putitem">PutItem</a></li>
 * <li><a href="#deleteitem">DeleteItem</a></li>
 * <li><a href="#updateitem">UpdateItem</a></li>
 * </ul></li>
 * <li><a href="#ecs">ECS</a>
 * <p>
 * <ul>
 * <li><a href="#runtask">RunTask</a>
 * <p>
 * <ul>
 * <li><a href="#ec2">EC2</a></li>
 * <li><a href="#fargate">Fargate</a></li>
 * </ul></li>
 * </ul></li>
 * <li><a href="#emr">EMR</a>
 * <p>
 * <ul>
 * <li><a href="#create-cluster">Create Cluster</a></li>
 * <li><a href="#termination-protection">Termination Protection</a></li>
 * <li><a href="#terminate-cluster">Terminate Cluster</a></li>
 * <li><a href="#add-step">Add Step</a></li>
 * <li><a href="#cancel-step">Cancel Step</a></li>
 * <li><a href="#modify-instance-fleet">Modify Instance Fleet</a></li>
 * <li><a href="#modify-instance-group">Modify Instance Group</a></li>
 * </ul></li>
 * <li><a href="#eks">EKS</a>
 * <p>
 * <ul>
 * <li><a href="#call">Call</a></li>
 * </ul></li>
 * <li><a href="#eventbridge">EventBridge</a>
 * <p>
 * <ul>
 * <li><a href="#put-events">Put Events</a></li>
 * </ul></li>
 * <li><a href="#glue">Glue</a></li>
 * <li><a href="#glue-databrew">Glue DataBrew</a></li>
 * <li><a href="#lambda">Lambda</a></li>
 * <li><a href="#sagemaker">SageMaker</a>
 * <p>
 * <ul>
 * <li><a href="#create-training-job">Create Training Job</a></li>
 * <li><a href="#create-transform-job">Create Transform Job</a></li>
 * <li><a href="#create-endpoint">Create Endpoint</a></li>
 * <li><a href="#create-endpoint-config">Create Endpoint Config</a></li>
 * <li><a href="#create-model">Create Model</a></li>
 * <li><a href="#update-endpoint">Update Endpoint</a></li>
 * </ul></li>
 * <li><a href="#sns">SNS</a></li>
 * <li><a href="#step-functions">Step Functions</a>
 * <p>
 * <ul>
 * <li><a href="#start-execution">Start Execution</a></li>
 * <li><a href="#invoke-activity">Invoke Activity</a></li>
 * </ul></li>
 * <li><a href="#sqs">SQS</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * <h2>Task</h2>
 * <p>
 * A Task state represents a single unit of work performed by a state machine. In the
 * CDK, the exact work to be done is determined by a class that implements <code>IStepFunctionsTask</code>.
 * <p>
 * AWS Step Functions <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-service-integrations.html">integrates</a> with some AWS services so that you can call API
 * actions, and coordinate executions directly from the Amazon States Language in
 * Step Functions. You can directly call and pass parameters to the APIs of those
 * services.
 * <p>
 * <h2>Paths</h2>
 * <p>
 * In the Amazon States Language, a <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html">path</a> is a string beginning with <code>$</code> that you
 * can use to identify components within JSON text.
 * <p>
 * Learn more about input and output processing in Step Functions <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html">here</a>
 * <p>
 * <h3>InputPath</h3>
 * <p>
 * Both <code>InputPath</code> and <code>Parameters</code> fields provide a way to manipulate JSON as it
 * moves through your workflow. AWS Step Functions applies the <code>InputPath</code> field first,
 * and then the <code>Parameters</code> field. You can first filter your raw input to a selection
 * you want using InputPath, and then apply Parameters to manipulate that input
 * further, or add new values. If you don't specify an <code>InputPath</code>, a default value
 * of <code>$</code> will be used.
 * <p>
 * The following example provides the field named <code>input</code> as the input to the <code>Task</code>
 * state that runs a Lambda function.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object submitJob = LambdaInvoke.Builder.create(this, "Invoke Handler")
 *         .lambdaFunction(fn)
 *         .inputPath("$.input")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>OutputPath</h3>
 * <p>
 * Tasks also allow you to select a portion of the state output to pass to the next
 * state. This enables you to filter out unwanted information, and pass only the
 * portion of the JSON that you care about. If you don't specify an <code>OutputPath</code>,
 * a default value of <code>$</code> will be used. This passes the entire JSON node to the next
 * state.
 * <p>
 * The <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseSyntax">response</a> from a Lambda function includes the response from the function
 * as well as other metadata.
 * <p>
 * The following example assigns the output from the Task to a field named <code>result</code>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object submitJob = LambdaInvoke.Builder.create(this, "Invoke Handler")
 *         .lambdaFunction(fn)
 *         .outputPath("$.Payload.result")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>ResultSelector</h3>
 * <p>
 * You can use <a href="https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector"><code>ResultSelector</code></a>
 * to manipulate the raw result of a Task, Map or Parallel state before it is
 * passed to <a href="###ResultPath"><code>ResultPath</code></a>. For service integrations, the raw
 * result contains metadata in addition to the response payload. You can use
 * ResultSelector to construct a JSON payload that becomes the effective result
 * using static values or references to the raw result or context object.
 * <p>
 * The following example extracts the output payload of a Lambda function Task and combines
 * it with some static values and the state name from the context object.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke Handler")
 *         .lambdaFunction(fn)
 *         .resultSelector(Map.of(
 *                 "lambdaOutput", sfn.JsonPath.stringAt("$.Payload"),
 *                 "invokeRequestId", sfn.JsonPath.stringAt("$.SdkResponseMetadata.RequestId"),
 *                 "staticValue", Map.of(
 *                         "foo", "bar"),
 *                 "stateName", sfn.JsonPath.stringAt("$.State.Name")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>ResultPath</h3>
 * <p>
 * The output of a state can be a copy of its input, the result it produces (for
 * example, output from a Task state’s Lambda function), or a combination of its
 * input and result. Use <a href="https://docs.aws.amazon.com/step-functions/latest/dg/input-output-resultpath.html"><code>ResultPath</code></a> to control which combination of these is
 * passed to the state output. If you don't specify an <code>ResultPath</code>, a default
 * value of <code>$</code> will be used.
 * <p>
 * The following example adds the item from calling DynamoDB's <code>getItem</code> API to the state
 * input and passes it to the next state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * DynamoPutItem.Builder.create(this, "PutItem")
 *         .item(Map.of(
 *                 "MessageId", tasks.DynamoAttributeValue.fromString("message-id")))
 *         .table(myTable)
 *         .resultPath("$.Item")
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ The <code>OutputPath</code> is computed after applying <code>ResultPath</code>. All service integrations
 * return metadata as part of their response. When using <code>ResultPath</code>, it's not possible to
 * merge a subset of the task output to the input.
 * <p>
 * <h2>Task parameters from the state JSON</h2>
 * <p>
 * Most tasks take parameters. Parameter values can either be static, supplied directly
 * in the workflow definition (by specifying their values), or a value available at runtime
 * in the state machine's execution (either as its input or an output of a prior state).
 * Parameter values available at runtime can be specified via the <code>JsonPath</code> class,
 * using methods such as <code>JsonPath.stringAt()</code>.
 * <p>
 * The following example provides the field named <code>input</code> as the input to the Lambda function
 * and invokes it asynchronously.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object submitJob = LambdaInvoke.Builder.create(this, "Invoke Handler")
 *         .lambdaFunction(fn)
 *         .payload(sfn.TaskInput.fromJsonPathAt("$.input"))
 *         .invocationType(tasks.LambdaInvocationType.getEVENT())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html">intrinsic functions</a> with <code>JsonPath.stringAt()</code>.
 * Here is an example of starting an Athena query that is dynamically created using the task input:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object startQueryExecutionJob = AthenaStartQueryExecution.Builder.create(this, "Athena Start Query")
 *         .queryString(sfn.JsonPath.stringAt("States.Format('select contacts where year={};', $.year)"))
 *         .queryExecutionContext(Map.of(
 *                 "databaseName", "interactions"))
 *         .resultConfiguration(Map.of(
 *                 "encryptionConfiguration", Map.of(
 *                         "encryptionOption", tasks.EncryptionOption.getS3_MANAGED()),
 *                 "outputLocation", Map.of(
 *                         "bucketName", "mybucket",
 *                         "objectKey", "myprefix")))
 *         .integrationPattern(sfn.IntegrationPattern.getRUN_JOB())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Each service integration has its own set of parameters that can be supplied.
 * <p>
 * <h2>Evaluate Expression</h2>
 * <p>
 * Use the <code>EvaluateExpression</code> to perform simple operations referencing state paths. The
 * <code>expression</code> referenced in the task will be evaluated in a Lambda function
 * (<code>eval()</code>). This allows you to not have to write Lambda code for simple operations.
 * <p>
 * Example: convert a wait time from milliseconds to seconds, concat this in a message and wait:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object convertToSeconds = EvaluateExpression.Builder.create(this, "Convert to seconds")
 *         .expression("$.waitMilliseconds / 1000")
 *         .resultPath("$.waitSeconds")
 *         .build();
 * 
 * Object createMessage = EvaluateExpression.Builder.create(this, "Create message")
 *         // Note: this is a string inside a string.
 *         .expression("`Now waiting ${$.waitSeconds} seconds...`")
 *         .runtime(lambda.Runtime.getNODEJS_14_X())
 *         .resultPath("$.message")
 *         .build();
 * 
 * Object publishMessage = SnsPublish.Builder.create(this, "Publish message")
 *         .topic(new Topic(this, "cool-topic"))
 *         .message(sfn.TaskInput.fromJsonPathAt("$.message"))
 *         .resultPath("$.sns")
 *         .build();
 * 
 * Object wait = Wait.Builder.create(this, "Wait")
 *         .time(sfn.WaitTime.secondsPath("$.waitSeconds"))
 *         .build();
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(convertToSeconds
 *             .next(createMessage)
 *             .next(publishMessage).next(wait))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>EvaluateExpression</code> supports a <code>runtime</code> prop to specify the Lambda
 * runtime to use to evaluate the expression. Currently, only runtimes
 * of the Node.js family are supported.
 * <p>
 * <h2>API Gateway</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html">API Gateway</a> through the service integration pattern.
 * <p>
 * HTTP APIs are designed for low-latency, cost-effective integrations with AWS services, including AWS Lambda, and HTTP endpoints.
 * HTTP APIs support OIDC and OAuth 2.0 authorization, and come with built-in support for CORS and automatic deployments.
 * Previous-generation REST APIs currently offer more features. More details can be found <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-vs-rest.html">here</a>.
 * <p>
 * <h3>Call REST API Endpoint</h3>
 * <p>
 * The <code>CallApiGatewayRestApiEndpoint</code> calls the REST API endpoint.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.stepfunctions.*;
 * import .*;
 * 
 * 
 * Object restApi = new RestApi(stack, "MyRestApi");
 * 
 * CallApiGatewayRestApiEndpoint invokeTask = new CallApiGatewayRestApiEndpoint(stack, "Call REST API", new CallApiGatewayRestApiEndpointProps()
 *         .api(restApi)
 *         .stageName("prod")
 *         .method(HttpMethod.getGET()));
 * </pre></blockquote>
 * <p>
 * <h3>Call HTTP API Endpoint</h3>
 * <p>
 * The <code>CallApiGatewayHttpApiEndpoint</code> calls the HTTP API endpoint.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.stepfunctions.*;
 * import .*;
 * 
 * 
 * Object httpApi = new HttpApi(stack, "MyHttpApi");
 * 
 * CallApiGatewayHttpApiEndpoint invokeTask = new CallApiGatewayHttpApiEndpoint(stack, "Call HTTP API", new CallApiGatewayHttpApiEndpointProps()
 *         .apiId(httpApi.getApiId())
 *         .apiStack(cdk.Stack.of(httpApi))
 *         .method(HttpMethod.getGET()));
 * </pre></blockquote>
 * <p>
 * <h2>Athena</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html">Athena</a> through the service integration pattern.
 * <p>
 * <h3>StartQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html">StartQueryExecution</a> API runs the SQL query statement.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object startQueryExecutionJob = AthenaStartQueryExecution.Builder.create(this, "Start Athena Query")
 *         .queryString(sfn.JsonPath.stringAt("$.queryString"))
 *         .queryExecutionContext(Map.of(
 *                 "databaseName", "mydatabase"))
 *         .resultConfiguration(Map.of(
 *                 "encryptionConfiguration", Map.of(
 *                         "encryptionOption", tasks.EncryptionOption.getS3_MANAGED()),
 *                 "outputLocation", Map.of(
 *                         "bucketName", "query-results-bucket",
 *                         "objectKey", "folder")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>GetQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html">GetQueryExecution</a> API gets information about a single execution of a query.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object getQueryExecutionJob = AthenaGetQueryExecution.Builder.create(this, "Get Query Execution")
 *         .queryExecutionId(sfn.JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>GetQueryResults</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html">GetQueryResults</a> API that streams the results of a single query execution specified by QueryExecutionId from S3.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object getQueryResultsJob = AthenaGetQueryResults.Builder.create(this, "Get Query Results")
 *         .queryExecutionId(sfn.JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>StopQueryExecution</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html">StopQueryExecution</a> API that stops a query execution.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object stopQueryExecutionJob = AthenaStopQueryExecution.Builder.create(this, "Stop Query Execution")
 *         .queryExecutionId(sfn.JsonPath.stringAt("$.QueryExecutionId"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Batch</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html">Batch</a> through the service integration pattern.
 * <p>
 * <h3>SubmitJob</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> API submits an AWS Batch job from a job definition.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * BatchSubmitJob task = new BatchSubmitJob(this, "Submit Job", new BatchSubmitJobProps()
 *         .jobDefinitionArn(batchJobDefinitionArn)
 *         .jobName("MyJob")
 *         .jobQueueArn(batchQueueArn));
 * </pre></blockquote>
 * <p>
 * <h2>CodeBuild</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html">CodeBuild</a> through the service integration pattern.
 * <p>
 * <h3>StartBuild</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html">StartBuild</a> starts a CodeBuild Project by Project Name.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.codebuild.*;
 * 
 * 
 * Project codebuildProject = new Project(this, "Project", new ProjectProps()
 *         .projectName("MyTestProject")
 *         .buildSpec(codebuild.BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", asList("echo \"Hello, CodeBuild!\"")))))));
 * 
 * CodeBuildStartBuild task = new CodeBuildStartBuild(this, "Task", new CodeBuildStartBuildProps()
 *         .project(codebuildProject)
 *         .integrationPattern(sfn.IntegrationPattern.getRUN_JOB())
 *         .environmentVariablesOverride(Map.of(
 *                 "ZONE", new BuildEnvironmentVariable()
 *                         .type(codebuild.BuildEnvironmentVariableType.getPLAINTEXT())
 *                         .value(sfn.JsonPath.stringAt("$.envVariables.zone")))));
 * </pre></blockquote>
 * <p>
 * <h2>DynamoDB</h2>
 * <p>
 * You can call DynamoDB APIs from a <code>Task</code> state.
 * Read more about calling DynamoDB APIs <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ddb.html">here</a>
 * <p>
 * <h3>GetItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html">GetItem</a> operation returns a set of attributes for the item with the given primary key.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * DynamoGetItem.Builder.create(this, "Get Item")
 *         .key(Map.of("messageId", tasks.DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>PutItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html">PutItem</a> operation creates a new item, or replaces an old item with a new item.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * DynamoPutItem.Builder.create(this, "PutItem")
 *         .item(Map.of(
 *                 "MessageId", tasks.DynamoAttributeValue.fromString("message-007"),
 *                 "Text", tasks.DynamoAttributeValue.fromString(sfn.JsonPath.stringAt("$.bar")),
 *                 "TotalCount", tasks.DynamoAttributeValue.fromNumber(10)))
 *         .table(myTable)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>DeleteItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html">DeleteItem</a> operation deletes a single item in a table by primary key.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * DynamoDeleteItem.Builder.create(this, "DeleteItem")
 *         .key(Map.of("MessageId", tasks.DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .resultPath(sfn.JsonPath.getDISCARD())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>UpdateItem</h3>
 * <p>
 * The <a href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html">UpdateItem</a> operation edits an existing item's attributes, or adds a new item
 * to the table if it does not already exist.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * DynamoUpdateItem.Builder.create(this, "UpdateItem")
 *         .key(Map.of(
 *                 "MessageId", tasks.DynamoAttributeValue.fromString("message-007")))
 *         .table(myTable)
 *         .expressionAttributeValues(Map.of(
 *                 ":val", tasks.DynamoAttributeValue.numberFromString(sfn.JsonPath.stringAt("$.Item.TotalCount.N")),
 *                 ":rand", tasks.DynamoAttributeValue.fromNumber(20)))
 *         .updateExpression("SET TotalCount = :val + :rand")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>ECS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html">ECS/Fargate</a> through the service integration pattern.
 * <p>
 * <h3>RunTask</h3>
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html">RunTask</a> starts a new task using the specified task definition.
 * <p>
 * <h4>EC2</h4>
 * <p>
 * The EC2 launch type allows you to run your containerized applications on a cluster
 * of Amazon EC2 instances that you manage.
 * <p>
 * When a task that uses the EC2 launch type is launched, Amazon ECS must determine where
 * to place the task based on the requirements specified in the task definition, such as
 * CPU and memory. Similarly, when you scale down the task count, Amazon ECS must determine
 * which tasks to terminate. You can apply task placement strategies and constraints to
 * customize how Amazon ECS places and terminates tasks. Learn more about <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html">task placement</a>
 * <p>
 * The latest ACTIVE revision of the passed task definition is used for running the task.
 * <p>
 * The following example runs a job from a task definition on EC2
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * 
 * IVpc vpc = ec2.Vpc.fromLookup(this, "Vpc", new VpcLookupOptions()
 *         .isDefault(true));
 * 
 * Cluster cluster = new Cluster(this, "Ec2Cluster", new ClusterProps().vpc(vpc));
 * cluster.addCapacity("DefaultAutoScalingGroup", new AddCapacityOptions()
 *         .instanceType(new InstanceType("t2.micro"))
 *         .vpcSubnets(new SubnetSelection().subnetType(ec2.SubnetType.getPUBLIC())));
 * 
 * TaskDefinition taskDefinition = new TaskDefinition(this, "TD", new TaskDefinitionProps()
 *         .compatibility(ecs.Compatibility.getEC2()));
 * 
 * taskDefinition.addContainer("TheContainer", new ContainerDefinitionOptions()
 *         .image(ecs.ContainerImage.fromRegistry("foo/bar"))
 *         .memoryLimitMiB(256));
 * 
 * EcsRunTask runTask = new EcsRunTask(this, "Run", new EcsRunTaskProps()
 *         .integrationPattern(sfn.IntegrationPattern.getRUN_JOB())
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .launchTarget(new EcsEc2LaunchTarget(new EcsEc2LaunchTargetOptions()
 *                 .placementStrategies(asList(ecs.PlacementStrategy.spreadAcrossInstances(), ecs.PlacementStrategy.packedByCpu(), ecs.PlacementStrategy.randomly()))
 *                 .placementConstraints(asList(ecs.PlacementConstraint.memberOf("blieptuut"))))));
 * </pre></blockquote>
 * <p>
 * <h4>Fargate</h4>
 * <p>
 * AWS Fargate is a serverless compute engine for containers that works with Amazon
 * Elastic Container Service (ECS). Fargate makes it easy for you to focus on building
 * your applications. Fargate removes the need to provision and manage servers, lets you
 * specify and pay for resources per application, and improves security through application
 * isolation by design. Learn more about <a href="https://aws.amazon.com/fargate/">Fargate</a>
 * <p>
 * The Fargate launch type allows you to run your containerized applications without the need
 * to provision and manage the backend infrastructure. Just register your task definition and
 * Fargate launches the container for you. The latest ACTIVE revision of the passed
 * task definition is used for running the task. Learn more about
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html">Fargate Versioning</a>
 * <p>
 * The following example runs a job from a task definition on Fargate
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * 
 * IVpc vpc = ec2.Vpc.fromLookup(this, "Vpc", new VpcLookupOptions()
 *         .isDefault(true));
 * 
 * Cluster cluster = new Cluster(this, "FargateCluster", new ClusterProps().vpc(vpc));
 * 
 * TaskDefinition taskDefinition = new TaskDefinition(this, "TD", new TaskDefinitionProps()
 *         .memoryMiB("512")
 *         .cpu("256")
 *         .compatibility(ecs.Compatibility.getFARGATE()));
 * 
 * ContainerDefinition containerDefinition = taskDefinition.addContainer("TheContainer", new ContainerDefinitionOptions()
 *         .image(ecs.ContainerImage.fromRegistry("foo/bar"))
 *         .memoryLimitMiB(256));
 * 
 * EcsRunTask runTask = new EcsRunTask(this, "RunFargate", new EcsRunTaskProps()
 *         .integrationPattern(sfn.IntegrationPattern.getRUN_JOB())
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .assignPublicIp(true)
 *         .containerOverrides(asList(new ContainerOverride()
 *                 .containerDefinition(containerDefinition)
 *                 .environment(asList(new TaskEnvironmentVariable().name("SOME_KEY").value(sfn.JsonPath.stringAt("$.SomeKey"))))))
 *         .launchTarget(new EcsFargateLaunchTarget()));
 * </pre></blockquote>
 * <p>
 * <h2>EMR</h2>
 * <p>
 * Step Functions supports Amazon EMR through the service integration pattern.
 * The service integration APIs correspond to Amazon EMR APIs but differ in the
 * parameters that are used.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Create Cluster</h3>
 * <p>
 * Creates and starts running a cluster (job flow).
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html"><code>runJobFlow</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Role clusterRole = new Role(this, "ClusterRole", new RoleProps()
 *         .assumedBy(new ServicePrincipal("ec2.amazonaws.com")));
 * 
 * Role serviceRole = new Role(this, "ServiceRole", new RoleProps()
 *         .assumedBy(new ServicePrincipal("elasticmapreduce.amazonaws.com")));
 * 
 * Role autoScalingRole = new Role(this, "AutoScalingRole", new RoleProps()
 *         .assumedBy(new ServicePrincipal("elasticmapreduce.amazonaws.com")));
 * 
 * autoScalingRole.assumeRolePolicy.addStatements(
 * new PolicyStatement(new PolicyStatementProps()
 *         .effect(iam.Effect.getALLOW())
 *         .principals(asList(
 *             new ServicePrincipal("application-autoscaling.amazonaws.com")))
 *         .actions(asList("sts:AssumeRole"))));
 * 
 * EmrCreateCluster.Builder.create(this, "Create Cluster")
 *         .instances(Map.of())
 *         .clusterRole(clusterRole)
 *         .name(sfn.TaskInput.fromJsonPathAt('$.ClusterName').getValue())
 *         .serviceRole(serviceRole)
 *         .autoScalingRole(autoScalingRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Termination Protection</h3>
 * <p>
 * Locks a cluster (job flow) so the EC2 instances in the cluster cannot be
 * terminated by user intervention, an API call, or a job-flow error.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html"><code>setTerminationProtection</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrSetClusterTerminationProtection.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .terminationProtected(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Terminate Cluster</h3>
 * <p>
 * Shuts down a cluster (job flow).
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html"><code>terminateJobFlows</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrTerminateCluster.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Add Step</h3>
 * <p>
 * Adds a new step to a running cluster.
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html"><code>addJobFlowSteps</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrAddStep.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .name("StepName")
 *         .jar("Jar")
 *         .actionOnFailure(tasks.ActionOnFailure.getCONTINUE())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cancel Step</h3>
 * <p>
 * Cancels a pending step in a running cluster.
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html"><code>cancelSteps</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrCancelStep.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .stepId("StepId")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Modify Instance Fleet</h3>
 * <p>
 * Modifies the target On-Demand and target Spot capacities for the instance
 * fleet with the specified InstanceFleetName.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html"><code>modifyInstanceFleet</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrModifyInstanceFleetByName.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .instanceFleetName("InstanceFleetName")
 *         .targetOnDemandCapacity(2)
 *         .targetSpotCapacity(0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Modify Instance Group</h3>
 * <p>
 * Modifies the number of nodes and configuration settings of an instance group.
 * <p>
 * Corresponds to the <a href="https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html"><code>modifyInstanceGroups</code></a> API in EMR.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * EmrModifyInstanceGroupByName.Builder.create(this, "Task")
 *         .clusterId("ClusterId")
 *         .instanceGroupName(sfn.JsonPath.stringAt("$.InstanceGroupName"))
 *         .instanceGroup(Map.of(
 *                 "instanceCount", 1))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>EKS</h2>
 * <p>
 * Step Functions supports Amazon EKS through the service integration pattern.
 * The service integration APIs correspond to Amazon EKS APIs.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Call</h3>
 * <p>
 * Read and write Kubernetes resource objects via a Kubernetes API endpoint.
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html"><code>call</code></a> API in Step Functions Connector.
 * <p>
 * The following code snippet includes a Task state that uses eks:call to list the pods.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.eks.*;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * import software.amazon.awscdk.services.stepfunctions.tasks.*;
 * 
 * 
 * Cluster myEksCluster = new Cluster(this, "my sample cluster", new ClusterProps()
 *         .version(eks.KubernetesVersion.getV1_18())
 *         .clusterName("myEksCluster"));
 * 
 * new EksCall(stack, "Call a EKS Endpoint", new EksCallProps()
 *         .cluster(myEksCluster)
 *         .httpMethod(MethodType.getGET())
 *         .httpPath("/api/v1/namespaces/default/pods"));
 * </pre></blockquote>
 * <p>
 * <h2>EventBridge</h2>
 * <p>
 * Step Functions supports Amazon EventBridge through the service integration pattern.
 * The service integration APIs correspond to Amazon EventBridge APIs.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html">Read more</a> about the differences when using these service integrations.
 * <p>
 * <h3>Put Events</h3>
 * <p>
 * Send events to an EventBridge bus.
 * Corresponds to the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html"><code>put-events</code></a> API in Step Functions Connector.
 * <p>
 * The following code snippet includes a Task state that uses events:putevents to send an event to the default bus.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * import software.amazon.awscdk.services.events.*;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * import software.amazon.awscdk.services.stepfunctions.tasks.*;
 * 
 * 
 * Object myEventBus = events.EventBus(stack, "EventBus", Map.of(
 *         "eventBusName", "MyEventBus1"));
 * 
 * new EventBridgePutEvents(stack, "Send an event to EventBridge", new EventBridgePutEventsProps()
 *         .entries(asList(new EventBridgePutEventsEntry()
 *                 .detail(sfn.TaskInput.fromObject(Map.of(
 *                         "Message", "Hello from Step Functions!")))
 *                 .eventBus(myEventBus)
 *                 .detailType("MessageFromStepFunctions")
 *                 .source("step.functions"))));
 * </pre></blockquote>
 * <p>
 * <h2>Glue</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html">AWS Glue</a> through the service integration pattern.
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun"><code>StartJobRun</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * GlueStartJobRun.Builder.create(this, "Task")
 *         .glueJobName("my-glue-job")
 *         .arguments(sfn.TaskInput.fromObject(Map.of(
 *                 "key", "value")))
 *         .timeout(cdk.Duration.minutes(30))
 *         .notifyDelayAfter(cdk.Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Glue DataBrew</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html">AWS Glue DataBrew</a> through the service integration pattern.
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html"><code>StartJobRun</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * GlueDataBrewStartJobRun.Builder.create(this, "Task")
 *         .name("databrew-job")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Lambda</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html">Invoke</a> a Lambda function.
 * <p>
 * You can specify the input to your Lambda function through the <code>payload</code> attribute.
 * By default, Step Functions invokes Lambda function with the state input (JSON path '$')
 * as the input.
 * <p>
 * The following snippet invokes a Lambda Function with the state input as the payload
 * by referencing the <code>$</code> path.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke with state input")
 *         .lambdaFunction(fn)
 *         .build();
 * </pre></blockquote>
 * <p>
 * When a function is invoked, the Lambda service sends  <a href="https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseElements">these response
 * elements</a>
 * back.
 * <p>
 * ⚠️ The response from the Lambda function is in an attribute called <code>Payload</code>
 * <p>
 * The following snippet invokes a Lambda Function by referencing the <code>$.Payload</code> path
 * to reference the output of a Lambda executed before it.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke with empty object as payload")
 *         .lambdaFunction(fn)
 *         .payload(sfn.TaskInput.fromObject(Map.of()))
 *         .build();
 * 
 * // use the output of fn as input
 * // use the output of fn as input
 * LambdaInvoke.Builder.create(this, "Invoke with payload field in the state input")
 *         .lambdaFunction(fn)
 *         .payload(sfn.TaskInput.fromJsonPathAt("$.Payload"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following snippet invokes a Lambda and sets the task output to only include
 * the Lambda function response.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke and set function response as task output")
 *         .lambdaFunction(fn)
 *         .outputPath("$.Payload")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to combine the input and the Lambda function response you can use
 * the <code>payloadResponseOnly</code> property and specify the <code>resultPath</code>. This will put the
 * Lambda function ARN directly in the "Resource" string, but it conflicts with the
 * integrationPattern, invocationType, clientContext, and qualifier properties.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke and combine function response with task input")
 *         .lambdaFunction(fn)
 *         .payloadResponseOnly(true)
 *         .resultPath("$.fn")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can have Step Functions pause a task, and wait for an external process to
 * return a task token. Read more about the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/callback-task-sample-sqs.html#call-back-lambda-example">callback pattern</a>
 * <p>
 * To use the callback pattern, set the <code>token</code> property on the task. Call the Step
 * Functions <code>SendTaskSuccess</code> or <code>SendTaskFailure</code> APIs with the token to
 * indicate that the task has completed and the state machine should resume execution.
 * <p>
 * The following snippet invokes a Lambda with the task token as part of the input
 * to the Lambda.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * LambdaInvoke.Builder.create(this, "Invoke with callback")
 *         .lambdaFunction(fn)
 *         .integrationPattern(sfn.IntegrationPattern.getWAIT_FOR_TASK_TOKEN())
 *         .payload(sfn.TaskInput.fromObject(Map.of(
 *                 "token", sfn.JsonPath.getTaskToken(),
 *                 "input", sfn.JsonPath.stringAt("$.someField"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ The task will pause until it receives that task token back with a <code>SendTaskSuccess</code> or <code>SendTaskFailure</code>
 * call. Learn more about <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">Callback with the Task
 * Token</a>.
 * <p>
 * AWS Lambda can occasionally experience transient service errors. In this case, invoking Lambda
 * results in a 500 error, such as <code>ServiceException</code>, <code>AWSLambdaException</code>, or <code>SdkClientException</code>.
 * As a best practice, the <code>LambdaInvoke</code> task will retry on those errors with an interval of 2 seconds,
 * a back-off rate of 2 and 6 maximum attempts. Set the <code>retryOnServiceExceptions</code> prop to <code>false</code> to
 * disable this behavior.
 * <p>
 * <h2>SageMaker</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html">AWS SageMaker</a> through the service integration pattern.
 * <p>
 * <h3>Create Training Job</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html"><code>CreateTrainingJob</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * SageMakerCreateTrainingJob.Builder.create(this, "TrainSagemaker")
 *         .trainingJobName(sfn.JsonPath.stringAt("$.JobName"))
 *         .algorithmSpecification(Map.of(
 *                 "algorithmName", "BlazingText",
 *                 "trainingInputMode", tasks.InputMode.getFILE()))
 *         .inputDataConfig(asList(Map.of(
 *                 "channelName", "train",
 *                 "dataSource", Map.of(
 *                         "s3DataSource", Map.of(
 *                                 "s3DataType", tasks.S3DataType.getS3_PREFIX(),
 *                                 "s3Location", tasks.S3Location.fromJsonExpression("$.S3Bucket"))))))
 *         .outputDataConfig(Map.of(
 *                 "s3OutputLocation", tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(this, "Bucket", "mybucket"), "myoutputpath")))
 *         .resourceConfig(Map.of(
 *                 "instanceCount", 1,
 *                 "instanceType", new InstanceType(JsonPath.stringAt("$.InstanceType")),
 *                 "volumeSize", cdk.Size.gibibytes(50)))// optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
 *         .stoppingCondition(Map.of(
 *                 "maxRuntime", cdk.Duration.hours(2)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Transform Job</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html"><code>CreateTransformJob</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SageMakerCreateTransformJob.Builder.create(this, "Batch Inference")
 *         .transformJobName("MyTransformJob")
 *         .modelName("MyModelName")
 *         .modelClientOptions(Map.of(
 *                 "invocationsMaxRetries", 3, // default is 0
 *                 "invocationsTimeout", cdk.Duration.minutes(5)))
 *         .transformInput(Map.of(
 *                 "transformDataSource", Map.of(
 *                         "s3DataSource", Map.of(
 *                                 "s3Uri", "s3://inputbucket/train",
 *                                 "s3DataType", tasks.S3DataType.getS3_PREFIX()))))
 *         .transformOutput(Map.of(
 *                 "s3OutputPath", "s3://outputbucket/TransformJobOutputPath"))
 *         .transformResources(Map.of(
 *                 "instanceCount", 1,
 *                 "instanceType", ec2.InstanceType.of(ec2.InstanceClass.getM4(), ec2.InstanceSize.getXLARGE())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Endpoint</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html"><code>CreateEndpoint</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SageMakerCreateEndpoint.Builder.create(this, "SagemakerEndpoint")
 *         .endpointName(sfn.JsonPath.stringAt("$.EndpointName"))
 *         .endpointConfigName(sfn.JsonPath.stringAt("$.EndpointConfigName"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Endpoint Config</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html"><code>CreateEndpointConfig</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SageMakerCreateEndpointConfig.Builder.create(this, "SagemakerEndpointConfig")
 *         .endpointConfigName("MyEndpointConfig")
 *         .productionVariants(asList(Map.of(
 *                 "initialInstanceCount", 2,
 *                 "instanceType", ec2.InstanceType.of(ec2.InstanceClass.getM5(), ec2.InstanceSize.getXLARGE()),
 *                 "modelName", "MyModel",
 *                 "variantName", "awesome-variant")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Create Model</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html"><code>CreateModel</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SageMakerCreateModel.Builder.create(this, "Sagemaker")
 *         .modelName("MyModel")
 *         .primaryContainer(ContainerDefinition.Builder.create()
 *                 .image(tasks.DockerImage.fromJsonExpression(sfn.JsonPath.stringAt("$.Model.imageName")))
 *                 .mode(tasks.Mode.getSINGLE_MODEL())
 *                 .modelS3Location(tasks.S3Location.fromJsonExpression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Update Endpoint</h3>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html"><code>UpdateEndpoint</code></a> API from a <code>Task</code> state.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * SageMakerUpdateEndpoint.Builder.create(this, "SagemakerEndpoint")
 *         .endpointName(sfn.JsonPath.stringAt("$.Endpoint.Name"))
 *         .endpointConfigName(sfn.JsonPath.stringAt("$.Endpoint.EndpointConfig"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SNS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sns.html">Amazon SNS</a> through the service integration pattern.
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/sns/latest/api/API_Publish.html"><code>Publish</code></a> API from a <code>Task</code> state to publish to an SNS topic.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
 * Topic topic = new Topic(this, "Topic");
 * 
 * // Use a field from the execution data as message.
 * Object task1 = SnsPublish.Builder.create(this, "Publish1")
 *         .topic(topic)
 *         .integrationPattern(sfn.IntegrationPattern.getREQUEST_RESPONSE())
 *         .message(sfn.TaskInput.fromDataAt("$.state.message"))
 *         .messageAttributes(Map.of(
 *                 "place", Map.of(
 *                         "value", sfn.JsonPath.stringAt("$.place")),
 *                 "pic", Map.of(
 *                         // BINARY must be explicitly set
 *                         "type", MessageAttributeDataType.getBINARY(),
 *                         "value", sfn.JsonPath.stringAt("$.pic")),
 *                 "people", Map.of(
 *                         "value", 4),
 *                 "handles", Map.of(
 *                         "value", asList("&#64;kslater", "&#64;jjf", null, "&#64;mfanning"))))
 *         .build();
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * Object task2 = SnsPublish.Builder.create(this, "Publish2")
 *         .topic(topic)
 *         .message(sfn.TaskInput.fromObject(Map.of(
 *                 "field1", "somedata",
 *                 "field2", sfn.JsonPath.stringAt("$.field2"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Step Functions</h2>
 * <p>
 * <h3>Start Execution</h3>
 * <p>
 * You can manage <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html">AWS Step Functions</a> executions.
 * <p>
 * AWS Step Functions supports it's own <a href="https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html"><code>StartExecution</code></a> API as a service integration.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * // Define a state machine with one Pass state
 * Object child = StateMachine.Builder.create(this, "ChildStateMachine")
 *         .definition(sfn.Chain.start(new Pass(this, "PassState")))
 *         .build();
 * 
 * // Include the state machine in a Task state with callback pattern
 * Object task = StepFunctionsStartExecution.Builder.create(this, "ChildTask")
 *         .stateMachine(child)
 *         .integrationPattern(sfn.IntegrationPattern.getWAIT_FOR_TASK_TOKEN())
 *         .input(sfn.TaskInput.fromObject(Map.of(
 *                 "token", sfn.JsonPath.getTaskToken(),
 *                 "foo", "bar")))
 *         .name("MyExecutionName")
 *         .build();
 * 
 * // Define a second state machine with the Task state above
 * // Define a second state machine with the Task state above
 * StateMachine.Builder.create(this, "ParentStateMachine")
 *         .definition(task)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Invoke Activity</h3>
 * <p>
 * You can invoke a <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html">Step Functions Activity</a> which enables you to have
 * a task in your state machine where the work is performed by a <em>worker</em> that can
 * be hosted on Amazon EC2, Amazon ECS, AWS Lambda, basically anywhere. Activities
 * are a way to associate code running somewhere (known as an activity worker) with
 * a specific task in a state machine.
 * <p>
 * When Step Functions reaches an activity task state, the workflow waits for an
 * activity worker to poll for a task. An activity worker polls Step Functions by
 * using GetActivityTask, and sending the ARN for the related activity.
 * <p>
 * After the activity worker completes its work, it can provide a report of its
 * success or failure by using <code>SendTaskSuccess</code> or <code>SendTaskFailure</code>. These two
 * calls use the taskToken provided by GetActivityTask to associate the result
 * with that task.
 * <p>
 * The following example creates an activity and creates a task that invokes the activity.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Object submitJobActivity = new Activity(this, "SubmitJob");
 * 
 * StepFunctionsInvokeActivity.Builder.create(this, "Submit Job")
 *         .activity(submitJobActivity)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>SQS</h2>
 * <p>
 * Step Functions supports <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-sqs.html">Amazon SQS</a>
 * <p>
 * You can call the <a href="https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html"><code>SendMessage</code></a> API from a <code>Task</code> state
 * to send a message to an SQS queue.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated. See https://github.com/aws/jsii/issues/826
 * Queue queue = new Queue(this, "Queue");
 * 
 * // Use a field from the execution data as message.
 * Object task1 = SqsSendMessage.Builder.create(this, "Send1")
 *         .queue(queue)
 *         .messageBody(sfn.TaskInput.fromJsonPathAt("$.message"))
 *         .build();
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * Object task2 = SqsSendMessage.Builder.create(this, "Send2")
 *         .queue(queue)
 *         .messageBody(sfn.TaskInput.fromObject(Map.of(
 *                 "field1", "somedata",
 *                 "field2", sfn.JsonPath.stringAt("$.field2"))))
 *         .build();
 * </pre></blockquote>
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Stable)
package software.amazon.awscdk.services.stepfunctions.tasks;
