/**
 * <h2>AWS Step Functions Construct Library</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p><strong>This is a <em>developer preview</em> (public beta) module. Releases might lack important features and might have
 * future breaking changes.</strong></p>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>The <code>@aws-cdk/aws-stepfunctions</code> package contains constructs for building
 * serverless workflows using objects. Use this in conjunction with the
 * <code>@aws-cdk/aws-stepfunctions-tasks</code> package, which contains classes used
 * to call other AWS services.</p>
 * <p>Defining a workflow looks like this (for the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/job-status-poller-sample.html">Step Functions Job Poller
 * example</a>):</p>
 * <h3>TypeScript example</h3>
 * <pre><code class="language-ts">import sfn = require('@aws-cdk/aws-stepfunctions');
 * import tasks = require('@aws-cdk/aws-stepfunctions-tasks');
 * 
 * const submitLambda = new lambda.Function(this, 'SubmitLambda', { ... });
 * const getStatusLambda = new lambda.Function(this, 'CheckLambda', { ... });
 * 
 * const submitJob = new sfn.Task(this, 'Submit Job', {
 *     task: new tasks.InvokeFunction(submitLambda),
 *     // Put Lambda's result here in the execution's state object
 *     resultPath: '$.guid',
 * });
 * 
 * const waitX = new sfn.Wait(this, 'Wait X Seconds', {
 *     duration: sfn.WaitDuration.secondsPath('$.wait_time'),
 * });
 * 
 * const getStatus = new sfn.Task(this, 'Get Job Status', {
 *     task: new tasks.InvokeFunction(getStatusLambda),
 *     // Pass just the field named &quot;guid&quot; into the Lambda, put the
 *     // Lambda's result in a field called &quot;status&quot;
 *     inputPath: '$.guid',
 *     resultPath: '$.status',
 * });
 * 
 * const jobFailed = new sfn.Fail(this, 'Job Failed', {
 *     cause: 'AWS Batch Job Failed',
 *     error: 'DescribeJob returned FAILED',
 * });
 * 
 * const finalStatus = new sfn.Task(this, 'Get Final Job Status', {
 *     task: new tasks.InvokeFunction(getStatusLambda),
 *     // Use &quot;guid&quot; field as input, output of the Lambda becomes the
 *     // entire state machine output.
 *     inputPath: '$.guid',
 * });
 * 
 * const definition = submitJob
 *     .next(waitX)
 *     .next(getStatus)
 *     .next(new sfn.Choice(this, 'Job Complete?')
 *         // Look at the &quot;status&quot; field
 *         .when(sfn.Condition.stringEquals('$.status', 'FAILED'), jobFailed)
 *         .when(sfn.Condition.stringEquals('$.status', 'SUCCEEDED'), finalStatus)
 *         .otherwise(waitX));
 * 
 * new sfn.StateMachine(this, 'StateMachine', {
 *     definition,
 *     timeoutSec: 300
 * });
 * </code></pre>
 * <h2>State Machine</h2>
 * <p>A <code>stepfunctions.StateMachine</code> is a resource that takes a state machine
 * definition. The definition is specified by its start state, and encompasses
 * all states reachable from the start state:</p>
 * <pre><code class="language-ts">const startState = new stepfunctions.Pass(this, 'StartState');
 * 
 * new stepfunctions.StateMachine(this, 'StateMachine', {
 *     definition: startState
 * });
 * </code></pre>
 * <p>State machines execute using an IAM Role, which will automatically have all
 * permissions added that are required to make all state machine tasks execute
 * properly (for example, permissions to invoke any Lambda functions you add to
 * your workflow). A role will be created by default, but you can supply an
 * existing one as well.</p>
 * <h2>Amazon States Language</h2>
 * <p>This library comes with a set of classes that model the <a href="https://states-language.net/spec.html">Amazon States
 * Language</a>. The following State classes
 * are supported:</p>
 * <ul>
 * <li><code>Task</code></li>
 * <li><code>Pass</code></li>
 * <li><code>Wait</code></li>
 * <li><code>Choice</code></li>
 * <li><code>Parallel</code></li>
 * <li><code>Succeed</code></li>
 * <li><code>Fail</code></li>
 * </ul>
 * <p>An arbitrary JSON object (specified at execution start) is passed from state to
 * state and transformed during the execution of the workflow. For more
 * information, see the States Language spec.</p>
 * <h3>Task</h3>
 * <p>A <code>Task</code> represents some work that needs to be done. The exact work to be
 * done is determine by a class that implements <code>IStepFunctionsTask</code>, a collection
 * of which can be found in the <code>@aws-cdk/aws-stepfunctions-tasks</code> package. A
 * couple of the tasks available are:</p>
 * <ul>
 * <li><code>tasks.InvokeFunction</code> -- call a Lambda Function</li>
 * <li><code>tasks.InvokeActivity</code> -- start an Activity (Activities represent a work
 * queue that you poll on a compute fleet you manage yourself)</li>
 * <li><code>tasks.PublishToTopic</code> -- publish a message to an SNS topic</li>
 * <li><code>tasks.SendToQueue</code> -- send a message to an SQS queue</li>
 * <li><code>tasks.RunEcsFargateTask</code>/<code>ecs.RunEcsEc2Task</code> -- run a container task,
 * depending on the type of capacity.</li>
 * <li><code>tasks.SagemakerTrainTask</code> -- run a SageMaker training job</li>
 * <li><code>tasks.SagemakerTransformTask</code> -- run a SageMaker transform job</li>
 * </ul>
 * <h4>Task parameters from the state json</h4>
 * <p>Many tasks take parameters. The values for those can either be supplied
 * directly in the workflow definition (by specifying their values), or at
 * runtime by passing a value obtained from the static functions on <code>Data</code>,
 * such as <code>Data.stringAt()</code>.</p>
 * <p>If so, the value is taken from the indicated location in the state JSON,
 * similar to (for example) <code>inputPath</code>.</p>
 * <h4>Lambda example</h4>
 * <pre><code class="language-ts">const task = new sfn.Task(this, 'Invoke The Lambda', {
 *     task: new tasks.InvokeFunction(myLambda),
 *     inputPath: '$.input',
 *     timeoutSeconds: 300,
 * });
 * 
 * // Add a retry policy
 * task.addRetry({
 *     intervalSeconds: 5,
 *     maxAttempts: 10
 * });
 * 
 * // Add an error handler
 * task.addCatch(errorHandlerState);
 * 
 * // Set the next state
 * task.next(nextState);
 * </code></pre>
 * <h4>SNS example</h4>
 * <pre><code class="language-ts">import sns = require('@aws-cdk/aws-sns');
 * 
 * // ...
 * 
 * const topic = new sns.Topic(this, 'Topic');
 * 
 * // Use a field from the execution data as message.
 * const task1 = new sfn.Task(this, 'Publish1', {
 *     task: new tasks.PublishToTopic(topic, {
 *         message: TaskInput.fromDataAt('$.state.message'),
 *     })
 * });
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * const task2 = new sfn.Task(this, 'Publish2', {
 *     task: new tasks.PublishToTopic(topic, {
 *         message: TaskInput.fromObject({
 *             field1: 'somedata',
 *             field2: Data.stringAt('$.field2'),
 *         })
 *     })
 * });
 * </code></pre>
 * <h4>SQS example</h4>
 * <pre><code class="language-ts">import sqs = require('@aws-cdk/aws-sqs');
 * 
 * // ...
 * 
 * const queue = new sns.Queue(this, 'Queue');
 * 
 * // Use a field from the execution data as message.
 * const task1 = new sfn.Task(this, 'Send1', {
 *     task: new tasks.SendToQueue(queue, {
 *         messageBody: TaskInput.fromDataAt('$.message'),
 *         // Only for FIFO queues
 *         messageGroupId: '1234'
 *     })
 * });
 * 
 * // Combine a field from the execution data with
 * // a literal object.
 * const task2 = new sfn.Task(this, 'Send2', {
 *     task: new tasks.SendToQueue(queue, {
 *         messageBody: TaskInput.fromObject({
 *             field1: 'somedata',
 *             field2: Data.stringAt('$.field2'),
 *         }),
 *         // Only for FIFO queues
 *         messageGroupId: '1234'
 *     })
 * });
 * </code></pre>
 * <h4>ECS example</h4>
 * <pre><code class="language-ts">import ecs = require('@aws-cdk/aws-ecs');
 * 
 * // See examples in ECS library for initialization of 'cluster' and 'taskDefinition'
 * 
 * const fargateTask = new ecs.RunEcsFargateTask({
 *   cluster,
 *   taskDefinition,
 *   containerOverrides: [
 *     {
 *       containerName: 'TheContainer',
 *       environment: [
 *         {
 *           name: 'CONTAINER_INPUT',
 *           value: Data.stringAt('$.valueFromStateData')
 *         }
 *       ]
 *     }
 *   ]
 * });
 * 
 * fargateTask.connections.allowToDefaultPort(rdsCluster, 'Read the database');
 * 
 * const task = new sfn.Task(this, 'CallFargate', {
 *     task: fargateTask
 * });
 * </code></pre>
 * <h4>SageMaker Transform example</h4>
 * <pre><code class="language-ts">const transformJob = new tasks.SagemakerTransformTask(
 *     transformJobName: &quot;MyTransformJob&quot;,
 *     modelName: &quot;MyModelName&quot;,
 *     role,
 *     transformInput: {
 *         transformDataSource: {
 *             s3DataSource: {
 *                 s3Uri: 's3://inputbucket/train',
 *                 s3DataType: S3DataType.S3Prefix,
 *             }
 *         }
 *     },
 *     transformOutput: {
 *         s3OutputPath: 's3://outputbucket/TransformJobOutputPath',
 *     },
 *     transformResources: {
 *         instanceCount: 1,
 *         instanceType: new ec2.InstanceTypePair(ec2.InstanceClass.M4, ec2.InstanceSize.XLarge),
 * });
 * 
 * const task = new sfn.Task(this, 'Batch Inference', {
 *     task: transformJob
 * });
 * </code></pre>
 * <h3>Pass</h3>
 * <p>A <code>Pass</code> state does no work, but it can optionally transform the execution's
 * JSON state.</p>
 * <pre><code class="language-ts">// Makes the current JSON state { ..., &quot;subObject&quot;: { &quot;hello&quot;: &quot;world&quot; } }
 * const pass = new stepfunctions.Pass(this, 'Add Hello World', {
 *     result: { hello: &quot;world&quot; },
 *     resultPath: '$.subObject',
 * });
 * 
 * // Set the next state
 * pass.next(nextState);
 * </code></pre>
 * <h3>Wait</h3>
 * <p>A <code>Wait</code> state waits for a given number of seconds, or until the current time
 * hits a particular time. The time to wait may be taken from the execution's JSON
 * state.</p>
 * <pre><code class="language-ts">// Wait until it's the time mentioned in the the state object's &quot;triggerTime&quot;
 * // field.
 * const wait = new stepfunctions.Wait(this, 'Wait For Trigger Time', {
 *     duration: stepfunctions.WaitDuration.timestampPath('$.triggerTime'),
 * });
 * 
 * // Set the next state
 * wait.next(startTheWork);
 * </code></pre>
 * <h3>Choice</h3>
 * <p>A <code>Choice</code> state can take a differen path through the workflow based on the
 * values in the execution's JSON state:</p>
 * <pre><code class="language-ts">const choice = new stepfunctions.Choice(this, 'Did it work?');
 * 
 * // Add conditions with .when()
 * choice.when(stepfunctions.Condition.stringEqual('$.status', 'SUCCESS'), successState);
 * choice.when(stepfunctions.Condition.numberGreaterThan('$.attempts', 5), failureState);
 * 
 * // Use .otherwise() to indicate what should be done if none of the conditions match
 * choice.otherwise(tryAgainState);
 * </code></pre>
 * <p>If you want to temporarily branch your workflow based on a condition, but have
 * all branches come together and continuing as one (similar to how an <code>if ... then ... else</code> works in a programming language), use the <code>.afterwards()</code> method:</p>
 * <pre><code class="language-ts">const choice = new stepfunctions.Choice(this, 'What color is it?');
 * choice.when(stepfunctions.Condition.stringEqual('$.color', 'BLUE'), handleBlueItem);
 * choice.when(stepfunctions.Condition.stringEqual('$.color', 'RED'), handleRedItem);
 * choice.otherwise(handleOtherItemColor);
 * 
 * // Use .afterwards() to join all possible paths back together and continue
 * choice.afterwards().next(shipTheItem);
 * </code></pre>
 * <p>If your <code>Choice</code> doesn't have an <code>otherwise()</code> and none of the conditions match
 * the JSON state, a <code>NoChoiceMatched</code> error will be thrown. Wrap the state machine
 * in a <code>Parallel</code> state if you want to catch and recover from this.</p>
 * <h3>Parallel</h3>
 * <p>A <code>Parallel</code> state executes one or more subworkflows in parallel. It can also
 * be used to catch and recover from errors in subworkflows.</p>
 * <pre><code class="language-ts">const parallel = new stepfunctions.Parallel(this, 'Do the work in parallel');
 * 
 * // Add branches to be executed in parallel
 * parallel.branch(shipItem);
 * parallel.branch(sendInvoice);
 * parallel.branch(restock);
 * 
 * // Retry the whole workflow if something goes wrong
 * parallel.addRetry({ maxAttempts: 1 });
 * 
 * // How to recover from errors
 * parallel.addCatch(sendFailureNotification);
 * 
 * // What to do in case everything succeeded
 * parallel.next(closeOrder);
 * </code></pre>
 * <h3>Succeed</h3>
 * <p>Reaching a <code>Succeed</code> state terminates the state machine execution with a
 * succesful status.</p>
 * <pre><code class="language-ts">const success = new stepfunctions.Succeed(this, 'We did it!');
 * </code></pre>
 * <h3>Fail</h3>
 * <p>Reaching a <code>Fail</code> state terminates the state machine execution with a
 * failure status. The fail state should report the reason for the failure.
 * Failures can be caught by encompassing <code>Parallel</code> states.</p>
 * <pre><code class="language-ts">const success = new stepfunctions.Fail(this, 'Fail', {
 *     error: 'WorkflowFailure',
 *     cause: &quot;Something went wrong&quot;
 * });
 * </code></pre>
 * <h2>Task Chaining</h2>
 * <p>To make defining work flows as convenient (and readable in a top-to-bottom way)
 * as writing regular programs, it is possible to chain most methods invocations.
 * In particular, the <code>.next()</code> method can be repeated. The result of a series of
 * <code>.next()</code> calls is called a <strong>Chain</strong>, and can be used when defining the jump
 * targets of <code>Choice.on</code> or <code>Parallel.branch</code>:</p>
 * <pre><code class="language-ts">const definition = step1
 *     .next(step2)
 *     .next(choice
 *         .when(condition1, step3.next(step4).next(step5))
 *         .otherwise(step6)
 *         .afterwards())
 *     .next(parallel
 *         .branch(step7.next(step8))
 *         .branch(step9.next(step10)))
 *     .next(finish);
 * 
 * new stepfunctions.StateMachine(this, 'StateMachine', {
 *     definition,
 * });
 * </code></pre>
 * <p>If you don't like the visual look of starting a chain directly off the first
 * step, you can use <code>Chain.start</code>:</p>
 * <pre><code class="language-ts">const definition = stepfunctions.Chain
 *     .start(step1)
 *     .next(step2)
 *     .next(step3)
 *     // ...
 * </code></pre>
 * <h2>State Machine Fragments</h2>
 * <p>It is possible to define reusable (or abstracted) mini-state machines by
 * defining a construct that implements <code>IChainable</code>, which requires you to define
 * two fields:</p>
 * <ul>
 * <li><code>startState: State</code>, representing the entry point into this state machine.</li>
 * <li><code>endStates: INextable[]</code>, representing the (one or more) states that outgoing
 * transitions will be added to if you chain onto the fragment.</li>
 * </ul>
 * <p>Since states will be named after their construct IDs, you may need to prefix the
 * IDs of states if you plan to instantiate the same state machine fragment
 * multiples times (otherwise all states in every instantiation would have the same
 * name).</p>
 * <p>The class <code>StateMachineFragment</code> contains some helper functions (like
 * <code>prefixStates()</code>) to make it easier for you to do this. If you define your state
 * machine as a subclass of this, it will be convenient to use:</p>
 * <pre><code class="language-ts">interface MyJobProps {
 *     jobFlavor: string;
 * }
 * 
 * class MyJob extends stepfunctions.StateMachineFragment {
 *     public readonly startState: State;
 *     public readonly endStates: INextable[];
 * 
 *     constructor(parent: cdk.Construct, id: string, props: MyJobProps) {
 *         super(parent, id);
 * 
 *         const first = new stepfunctions.Task(this, 'First', { ... });
 *         // ...
 *         const last = new stepfunctions.Task(this, 'Last', { ... });
 * 
 *         this.startState = first;
 *         this.endStates = [last];
 *     }
 * }
 * 
 * // Do 3 different variants of MyJob in parallel
 * new stepfunctions.Parallel(this, 'All jobs')
 *     .branch(new MyJob(this, 'Quick', { jobFlavor: 'quick' }).prefixStates())
 *     .branch(new MyJob(this, 'Medium', { jobFlavor: 'medium' }).prefixStates())
 *     .branch(new MyJob(this, 'Slow', { jobFlavor: 'slow' }).prefixStates());
 * </code></pre>
 * <h2>Activity</h2>
 * <p><strong>Activities</strong> represent work that is done on some non-Lambda worker pool. The
 * Step Functions workflow will submit work to this Activity, and a worker pool
 * that you run yourself, probably on EC2, will pull jobs from the Activity and
 * submit the results of individual jobs back.</p>
 * <p>You need the ARN to do so, so if you use Activities be sure to pass the Activity
 * ARN into your worker pool:</p>
 * <pre><code class="language-ts">const activity = new stepfunctions.Activity(this, 'Activity');
 * 
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * new cdk.CfnOutput(this, 'ActivityArn', { value: activity.activityArn });
 * </code></pre>
 * <h2>Metrics</h2>
 * <p><code>Task</code> object expose various metrics on the execution of that particular task. For example,
 * to create an alarm on a particular task failing:</p>
 * <pre><code class="language-ts">new cloudwatch.Alarm(this, 'TaskAlarm', {
 *     metric: task.metricFailed(),
 *     threshold: 1,
 *     evaluationPeriods: 1,
 * });
 * </code></pre>
 * <p>There are also metrics on the complete state machine:</p>
 * <pre><code class="language-ts">new cloudwatch.Alarm(this, 'StateMachineAlarm', {
 *     metric: stateMachine.metricFailed(),
 *     threshold: 1,
 *     evaluationPeriods: 1,
 * });
 * </code></pre>
 * <p>And there are metrics on the capacity of all state machines in your account:</p>
 * <pre><code class="language-ts">new cloudwatch.Alarm(this, 'ThrottledAlarm', {
 *     metric: StateTransitionMetrics.metricThrottledEvents(),
 *     threshold: 10,
 *     evaluationPeriods: 2,
 * });
 * </code></pre>
 * <h2>Future work</h2>
 * <p>Contributions welcome:</p>
 * <ul>
 * <li>[ ] A single <code>LambdaTask</code> class that is both a <code>Lambda</code> and a <code>Task</code> in one
 * might make for a nice API.</li>
 * <li>[ ] Expression parser for Conditions.</li>
 * <li>[ ] Simulate state machines in unit tests.</li>
 * </ul>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.stepfunctions;
