package tech.amwal.justpassme

import android.app.Activity

import kotlinx.coroutines.runBlocking
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import tech.amwal.justpassme.http.WebAuthnHttp
import tech.amwal.justpassme.proxy.WebAuthnProxy

/**
 * AndroidAuth is an authentication library that provides methods to register and authenticate users.
 *
 * @param activity The activity context.
 * @param clientUrl The client URL.
 * @param serviceUrl The service URL.
 * @param sessionId The session ID.
 */
class AndroidAuth(
    activity: Activity,
    clientUrl: String,
    serviceUrl: String,
    sessionId: String
) {
    private val httpClient = OkHttpClient.Builder()
        .addInterceptor(HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        })
        .build()

    private val repository = WebauthnRepository(
        webAuthnProxy = WebAuthnProxy(activity = activity),
        webAuthnHttp = WebAuthnHttp(
            client = httpClient,
            clientUrl = clientUrl,
            serviceUrl = serviceUrl
        ),
        sessionId = sessionId
    )

    /**
     * Registers a user and returns a map of the response.
     *
     * @return A map of the response.
     */
    fun register(): Map<String, *> {
        return runBlocking() {
            repository.register()
        }
    }

    /**
     * Authenticates a user and returns a map of the response.
     *
     * @return A map of the response.
     */
    fun auth(): Map<String, *> {
        return runBlocking() {
            repository.auth()
        }
    }

    /**
     * Suspends the registration process and returns a map of the response.
     *
     * @return A map of the response.
     */
    suspend fun suspendRegister(): Map<String, *> {
        return repository.register()
    }

    /**
     * Suspends the authentication process and returns a map of the response.
     *
     * @return A map of the response.
     */
    suspend fun suspendAuth(): Map<String, *> {
        return repository.auth()
    }
}