package tech.amwal.justpassme.http

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.Headers
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody.Companion.toRequestBody
import okhttp3.executeAsync
import tech.amwal.justpassme.model.PublicKeyWrapper
import tech.amwal.justpassme.model.auth.AuthenticationPublicKey
import tech.amwal.justpassme.model.auth.PublicKeyAssertionWrapper
import tech.amwal.justpassme.model.register.RegistrationPublicKey
import tech.amwal.justpassme.model.register.RegistrationVerifyResponseApi

internal class WebAuthnHttp(
    val client: OkHttpClient,
    private val clientUrl: String,
    private val serviceUrl: String
) {
    private val jsonContent = "application/json; charset=utf-8".toMediaType()
    internal suspend fun getPublicKey(sessionId:String): PublicKeyWrapper {
        val request = Request.Builder()
            .url(
                (clientUrl + PUBLIC_KEY_PATH).toHttpUrl()
            )
            .addHeader("AMWAL-PLATFORM", "app")
            .addHeader("dsessionId",sessionId)
            .get()
            .build()
        return client.newCall(request).executeAsync().use { response ->
            withContext(Dispatchers.IO) {
                val stringBody = response.body.string()
                Json.decodeFromString(stringBody)
            }
        }
    }

    internal suspend fun getPublicKeyAuth(sessionId:String): PublicKeyAssertionWrapper {
        val request = Request.Builder()
            .url(
                (clientUrl + PUBLIC_KEY_PATH).toHttpUrl()
            )
            .addHeader("AMWAL-PLATFORM", "app")
            .addHeader("dsessionId",sessionId)
            .get()
            .build()
        return client.newCall(request).executeAsync().use { response ->
            withContext(Dispatchers.IO) {
                val stringBody = response.body.string()
                Json.decodeFromString(stringBody)
            }
        }
    }

    internal suspend fun verifyRegistration(
        registrationPublicKey: RegistrationPublicKey,
        sessionId:String
    ): RegistrationVerifyResponseApi {
        val requestBody = Json.encodeToString(registrationPublicKey)
        val request = Request.Builder()
            .url(
                (serviceUrl + VERIFY_REGISTRATION_PATH).toHttpUrl()
            )
            .addHeader("dsessionId",sessionId)
            .post(requestBody.toRequestBody(contentType = jsonContent))
            .build()
        return client.newCall(request).executeAsync().use { response ->
            withContext(Dispatchers.IO) {
                val stringBody = response.body.string()
                Json.decodeFromString<RegistrationVerifyResponseApi>(stringBody)
            }
        }
    }

    internal suspend fun auth(
        authenticationPublicKey: AuthenticationPublicKey,
        sessionId:String
    ): RegistrationVerifyResponseApi {
        val requestBody = Json.encodeToString(authenticationPublicKey)
        val request = Request.Builder()
            .url(
                (serviceUrl + VERIFY_AUTH_PATH).toHttpUrl()

            )
            .addHeader("dsessionId",sessionId)
            .post(requestBody.toRequestBody(contentType = jsonContent))
            .build()
        return client.newCall(request).executeAsync().use { response ->
            withContext(Dispatchers.IO) {
                val stringBody = response.body.string()
                Json.decodeFromString(stringBody)
            }
        }
    }

    internal suspend fun backToClient(sessionId:String): Map<String, String> {
        val request = Request.Builder()
            .url(
                (serviceUrl + BACK_TO_CLIENT_PATH).toHttpUrl()
            )
            .get()
            .addHeader("dsessionId",sessionId)
            .build()
        return client.newCall(request).executeAsync().use { response ->
            withContext(Dispatchers.IO) {
                val stringBody = response.body.string()
                Json.decodeFromString(stringBody)
            }
        }
    }
    private fun Map<String, *>.asHttpHeaders(): Headers {
        return Headers.Builder().apply {
            forEach { (key, value) ->
                add(key, value.toString())
            }
        }
            .build()
    }

    companion object {
        internal const val PUBLIC_KEY_PATH = "/oidc/authenticate/"
        const val VERIFY_REGISTRATION_PATH = "/fido2/reg_complete/"
        const val VERIFY_AUTH_PATH = "/fido2/complete_auth/"
        const val BACK_TO_CLIENT_PATH = "/back_to_client/"
    }
}



