package tech.amwal.justpassme.proxy

import android.app.Activity
import android.util.Base64
import androidx.activity.ComponentActivity
import androidx.activity.result.ActivityResult
import androidx.activity.result.IntentSenderRequest
import androidx.activity.result.contract.ActivityResultContracts
import com.google.android.gms.fido.Fido
import com.google.android.gms.fido.fido2.api.common.AuthenticatorAssertionResponse
import com.google.android.gms.fido.fido2.api.common.AuthenticatorAttestationResponse
import com.google.android.gms.fido.fido2.api.common.AuthenticatorErrorResponse
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredential
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialCreationOptions
import com.google.android.gms.fido.fido2.api.common.PublicKeyCredentialRequestOptions
import kotlinx.coroutines.CancellableContinuation
import kotlinx.coroutines.suspendCancellableCoroutine
import kotlinx.coroutines.tasks.await
import tech.amwal.justpassme.model.auth.AuthenticationPublicKey
import tech.amwal.justpassme.model.register.RegistrationPublicKey
import kotlin.coroutines.resume

internal class WebAuthnProxy(private val activity: Activity) {
    private val fidoClient = Fido.getFido2ApiClient(activity)

    internal suspend fun registerUser(creationOptions: PublicKeyCredentialCreationOptions): RegistrationPublicKey {
        val intent = fidoClient.getRegisterPendingIntent(
            creationOptions
        ).await()
        return suspendCancellableCoroutine { continuation ->
            val launcher = (activity as ComponentActivity)
                .activityResultRegistry
                .register(
                    REGISTRATION_CODE,
                    ActivityResultContracts.StartIntentSenderForResult(),
                    handleRegistrationResult(continuation)
                )
            launcher.launch(
                IntentSenderRequest.Builder(intent).build()
            )
            continuation.invokeOnCancellation {
                launcher.unregister()
            }
        }
    }

    private fun handleRegistrationResult(
        continuation: CancellableContinuation<RegistrationPublicKey>
    ): (result: ActivityResult) -> Unit = { result ->
        val bytes = result.data?.getByteArrayExtra(Fido.FIDO2_KEY_CREDENTIAL_EXTRA)
        when {
            result.resultCode != Activity.RESULT_OK ->
                continuation.cancel(Throwable("Registration Canceled"))

            bytes == null ->
                continuation.cancel(Throwable("Error with the credentials"))

            else -> {
                val credential = PublicKeyCredential.deserializeFromBytes(bytes)
                val response = credential.response
                if (response is AuthenticatorErrorResponse) {
                    continuation.cancel(Throwable(response.toString()))
                } else {
                    val attestationResponse =
                        credential.response as AuthenticatorAttestationResponse
                    val id = credential.rawId.toBase64()

                    continuation.resume(
                        RegistrationPublicKey(
                            id = id,
                            rawId = id,
                            response = RegistrationPublicKey.Response(
                                client = response.clientDataJSON.toBase64(),
                                attestation = attestationResponse.attestationObject.toBase64()
                            )
                        )
                    )
                }
            }
        }
    }

    internal suspend fun signInUser(publicKey: PublicKeyCredentialRequestOptions): AuthenticationPublicKey {
        val intent = fidoClient.getSignPendingIntent(
            publicKey
        ).await()
        return suspendCancellableCoroutine { continuation ->
            val launcher = (activity as ComponentActivity)
                .activityResultRegistry
                .register(
                    SIGN_IN_CODE,
                    ActivityResultContracts.StartIntentSenderForResult(),
                    handleSignInResult(continuation)
                )
            launcher.launch(
                IntentSenderRequest.Builder(intent).build()
            )
            continuation.invokeOnCancellation {
                launcher.unregister()
            }
        }
    }

    private fun handleSignInResult(
        continuation: CancellableContinuation<AuthenticationPublicKey>
    ): (result: ActivityResult) -> Unit = { result: ActivityResult ->
        val bytes = result.data?.getByteArrayExtra(Fido.FIDO2_KEY_CREDENTIAL_EXTRA)
        when {
            result.resultCode != Activity.RESULT_OK ->
                continuation.cancel(Throwable("Registration Canceled"))

            bytes == null ->
                continuation.cancel(Throwable("Error Authentication"))

            else -> {
                val credential = PublicKeyCredential.deserializeFromBytes(bytes)
                val response = credential.response
                val assertionResponse = credential.response as AuthenticatorAssertionResponse

                if (response is AuthenticatorErrorResponse) {
                    continuation.cancel(Throwable(response.toString()))
                } else {
                    val id = credential.rawId.toBase64()
                    continuation.resume(
                        AuthenticationPublicKey(
                            id = id,
                            rawId = id,
                            response = AuthenticationPublicKey.Response(
                                signature = assertionResponse.signature.toBase64(),
                                userHandle = assertionResponse.userHandle?.toBase64(),
                                assertionDataJSON = assertionResponse.clientDataJSON.toBase64(),
                                authenticatorData = assertionResponse.authenticatorData.toBase64()
                            )
                        )
                    )
                }
            }
        }
    }

    companion object {
        private const val REGISTRATION_CODE = "registerUserCode"
        private const val SIGN_IN_CODE = "SignInUserCode"
        private const val BASE64_FLAG = Base64.NO_PADDING or Base64.NO_WRAP or Base64.URL_SAFE

        fun ByteArray.toBase64(): String {
            return Base64.encodeToString(this, BASE64_FLAG)
        }

        fun String.decodeBase64(): ByteArray {
            return Base64.decode(this, BASE64_FLAG)
        }
    }
}
