package tech.amwal.launcher.presentation

import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.filterIsInstance
import kotlinx.coroutines.flow.flatMapMerge
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEmpty
import kotlinx.coroutines.flow.onStart
import tech.amwal.components.presentation.ViewModel
import tech.amwal.phone.domain.usecase.SavePhoneNumber
import tech.amwal.transaction.domain.usecase.CreateTransaction
import tech.amwal.transaction.domain.usecase.GetSessionId
import tech.amwal.user.domain.GetUserAuthStateInteractor
import tech.amwal.user.domain.model.UserAuthState

@OptIn(FlowPreview::class)
class LauncherViewModel(
    private val createTransaction: CreateTransaction,
    private val getSessionId: GetSessionId,
    private val getUserAuthStateInteractor: GetUserAuthStateInteractor,
    private val savePhoneNumber: SavePhoneNumber
) :
    ViewModel<LauncherActions, LauncherResults, LauncherState, LauncherSideEffects>(LauncherState.Idle) {

    override suspend fun LauncherActions.process(): Flow<LauncherResults> =
        when (this) {
            is LauncherActions.StartPayment -> {
                createTransaction(amount, tax)
                    .flatMapMerge { getSessionId() }
                    .flatMapMerge { getUserAuthStateInteractor(phoneNumber, countryCode) }
                    .map { userState -> userState.asLauncherResult() }
                    // TODO("Start Auth Process here")
                    .filterIsInstance<LauncherResults>()
                    .onEmpty { emit(LauncherResults.PhoneNumberRequired) }
                    .catch { it.printStackTrace(); emit(LauncherResults.Error(it)) }
                    .onStart {
                        savePhoneNumber(phoneNumber, countryCode)
                    }
                    .onStart { emit(LauncherResults.InFlight) }
            }
        }

    override fun LauncherState.plus(result: LauncherResults): LauncherState = when (result) {
        LauncherResults.InFlight -> LauncherState.Loading
        is LauncherResults.Error -> LauncherState.Error(result.error)
        else -> LauncherState.Idle
    }

    override fun LauncherResults.effect(): LauncherSideEffects =
        when (this) {
            LauncherResults.InFlight -> LauncherSideEffects.Idle
            LauncherResults.PaymentMethodMissing -> LauncherSideEffects.AddRequiredPaymentMethod
            is LauncherResults.OtpAndBiometricsRegistrationRequired ->
                LauncherSideEffects.VerifyOtpAndRegisterBiometrics(phoneNumber)

            LauncherResults.AuthBiometrics -> LauncherSideEffects.AuthBiometrics
            LauncherResults.PhoneNumberRequired -> LauncherSideEffects.RegisterUserWithPhoneNumber
            is LauncherResults.Error -> LauncherSideEffects.Idle
        }

    private fun UserAuthState.asLauncherResult() = when (this) {
        UserAuthState.Empty, UserAuthState.RequiresPhoneNumber -> LauncherResults.PhoneNumberRequired
        UserAuthState.RequiresBiometricAuth -> LauncherResults.AuthBiometrics
        is UserAuthState.RequiresOtpAndBiometricRegistration ->
            LauncherResults.OtpAndBiometricsRegistrationRequired(
                phoneNumber
            )

        UserAuthState.RequiresPaymentMethod -> LauncherResults.PaymentMethodMissing
    }
}
