package tech.mhuang.pacebox.springboot.core.spring.start;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import tech.mhuang.pacebox.core.util.CollectionUtil;

import java.util.Map;

/**
 * spring启动装载类
 *
 * @author mhuang
 * @since 1.0.0
 */
public class SpringContextHolder implements ApplicationContextAware {

    private static ApplicationContext applicationContext;

    private static DefaultListableBeanFactory beanFactory;

    /**
     * 获取上下文
     *
     * @return ApplicationContext 上下文
     */
    public static ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    /**
     * 上下文初始化
     *
     * @param applicationContext 传递的上下文
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        if (SpringContextHolder.applicationContext == null) {
            SpringContextHolder.applicationContext = applicationContext;
            beanFactory = (DefaultListableBeanFactory) applicationContext.getAutowireCapableBeanFactory();
        }
    }

    /**
     * 通过名字获取上下文中的bean
     *
     * @param name bean方法名
     * @return Object 返回对应bean
     */
    public static Object getBean(String name) {
        return applicationContext.getBean(name);
    }

    /**
     * 通过名字获取上下文中的bean
     *
     * @param name  bean方法名
     * @param clazz 获取Bean指定的class
     * @param <T>   获取Bean指定的Class类型
     * @return BeanName的Bean
     */

    public static <T> T getBean(String name, Class<T> clazz) {
        return applicationContext.getBean(name, clazz);
    }

    /**
     * 通过名字删除上下文的bean
     *
     * @param name 删除的BeanName
     */
    public static void removeBean(String name) {
        beanFactory.removeBeanDefinition(name);
    }

    /**
     * bean注册
     *
     * @param beanName 注册的Bean Name
     * @param clazz    注册的Bean class
     * @param <T>      注册的Bean Class类型
     * @return 返回注册的Bean
     */
    public static <T> T registerBean(String beanName, Class<T> clazz) {
        return registerBean(beanName, clazz, null, new Object[0]);
    }

    /**
     * bean注册
     *
     * @param beanName 注册的Bean Name
     * @param clazz    注册的Bean class
     * @param <T>      注册的Bean Class类型
     * @param params   注册的Bean 传递的参数
     * @param args     注册的bean 类参数
     * @return 返回注册的Bean
     */
    public static <T> T registerBean(String beanName, Class<T> clazz, Map<String, Object> params, Object... args) {
        if (applicationContext.containsBean(beanName)) {
            Object bean = applicationContext.getBean(beanName);
            if (bean.getClass().isAssignableFrom(clazz)) {
                return (T) bean;
            } else {
                throw new RuntimeException("BeanName 重复 " + beanName);
            }
        }
        BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder.genericBeanDefinition(clazz);
        if (CollectionUtil.isNotEmpty(params)) {
            params.forEach((key, value) -> beanDefinitionBuilder.addPropertyValue(key, value));
        }
        if (CollectionUtil.isNotEmpty(args)) {
            for (Object arg : args) {
                beanDefinitionBuilder.addConstructorArgValue(arg);
            }
        }
        BeanDefinition beanDefinition = beanDefinitionBuilder.getRawBeanDefinition();
        BeanDefinitionRegistry beanFactory = (BeanDefinitionRegistry) ((ConfigurableApplicationContext) applicationContext).getBeanFactory();
        beanFactory.registerBeanDefinition(beanName, beanDefinition);
        return getBean(beanName, clazz);
    }

    /**
     * 根据class获取上下文的bean
     *
     * @param requiredType 获取Bean的Class
     * @param <T>          获取的class类型
     * @return Object 返回对应Bean
     */
    public static <T> T getBean(Class<T> requiredType) {
        return applicationContext.getBean(requiredType);
    }

    /**
     * 根据type获取bean相关配置
     *
     * @param type class
     * @param <T>  class类型
     * @return 所有集合
     * @since 2020.0.2.2
     */
    public static <T> Map<String, T> getBeansOfType(Class<T> type) {
        return applicationContext.getBeansOfType(type);
    }
}