/*
 * Copyright 2015 SirWellington Tech.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package tech.sirwellington.alchemy.generator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.sirwellington.alchemy.annotations.access.NonInstantiable;
import tech.sirwellington.alchemy.annotations.arguments.NonNull;
import tech.sirwellington.alchemy.annotations.designs.patterns.StrategyPattern;

import static tech.sirwellington.alchemy.annotations.designs.patterns.StrategyPattern.Role.CONCRETE_BEHAVIOR;
import static tech.sirwellington.alchemy.generator.AlchemyGenerator.one;
import static tech.sirwellington.alchemy.generator.Checks.checkNotNull;
import static tech.sirwellington.alchemy.generator.Checks.checkThat;
import static tech.sirwellington.alchemy.generator.NumberGenerators.integers;

/**
 * {@linkplain AlchemyGenerator Alchemy Generators} for Java Collections. These generators are useful in conjuction with
 * other {@linkplain AlchemyGenerator Generators}, such as those in {@link NumberGenerators} or
 * {@link StringGenerators}.
 *
 * @see StringGenerators
 * @see NumberGenerators
 * @see BinaryGenerators
 *
 * @author SirWellington
 */
@NonInstantiable
@StrategyPattern(role = CONCRETE_BEHAVIOR)
public final class CollectionGenerators
{

    private final static Logger LOG = LoggerFactory.getLogger(CollectionGenerators.class);

    CollectionGenerators() throws IllegalAccessException
    {
        throw new IllegalAccessException("cannot instantiate this class");
    }

    /**
     * Returns a list of Objects of varying size, using the supplied generator.
     *
     * @param <T>
     * @param generator The generator that produces values
     *
     * @return A list of random values, the length of which will vary.
     */
    public static <T> List<T> listOf(@NonNull AlchemyGenerator<T> generator)
    {
        return listOf(generator, one(integers(5, 200)));
    }

    /**
     * Returns a list of Objects using the supplied generator.
     *
     * @param <T>       The type to generate
     * @param generator The generator that produces values
     * @param size      The size of the list
     *
     * @return A list of random values with length {code size}
     *
     * @throws IllegalArgumentException if size is less than 1.
     */
    public static <T> List<T> listOf(@NonNull AlchemyGenerator<T> generator, int size)
    {
        checkThat(size > 0, "Size must be at least 1");
        checkNotNull(generator, "generator is null");
        List<T> list = new ArrayList<>(size);
        for (int i = 0; i < size; ++i)
        {
            list.add(generator.get());
        }
        return list;
    }

    /**
     * An {@link AlchemyGenerator} that returns values from a fixed list.
     *
     * @param <T>
     * @param list
     *
     * @throws IllegalArgumentException
     */
    public static <T> AlchemyGenerator<T> fromList(@NonNull List<T> list) throws IllegalArgumentException
    {
        checkNotNull(list, "list cannot be null");
        checkThat(!list.isEmpty(), "list has no elements");

        return () ->
        {
            int index = one(integers(0, list.size()));
            return list.get(index);
        };

    }

    /**
     * Convenience method for 
     * {@link #mapOf(tech.sirwellington.alchemy.generator.AlchemyGenerator, tech.sirwellington.alchemy.generator.AlchemyGenerator, int) }.
     * Maps returned will vary in size.
     *
     * @param <K>
     * @param <V>
     * @param keyGenerator
     * @param valueGenerator
     *
     * @return
     * @throws IllegalArgumentException
     */
    public static <K, V> Map<K, V> mapOf(@NonNull AlchemyGenerator<K> keyGenerator, @NonNull AlchemyGenerator<V> valueGenerator) throws IllegalArgumentException
    {
        int size = one(integers(5, 100));
        return mapOf(keyGenerator, valueGenerator, size);
    }

    /**
     * Creates a Map using the Keys and Values generated by the provided Generators.
     *
     * @param <K>            The Type of the Keys
     * @param <V>            The Type of the Values
     *
     * @param keyGenerator   Generates keys for the Map
     * @param valueGenerator Generates values for the Map
     * @param size           The exact size of the created map
     *
     * @return Map generated from the parameters specified.Ï
     * @throws IllegalArgumentException
     */
    public static <K, V> Map<K, V> mapOf(@NonNull AlchemyGenerator<K> keyGenerator, @NonNull AlchemyGenerator<V> valueGenerator, int size) throws IllegalArgumentException
    {
        checkThat(size > 0, "size must be at least 1");
        checkNotNull(keyGenerator);
        checkNotNull(valueGenerator);

        Map<K, V> map = new HashMap<>(size);

        for (int i = 0; i < size; ++i)
        {
            K key = keyGenerator.get();
            V vaue = valueGenerator.get();
            map.put(key, vaue);
        }

        return map;
    }

    //TODO:  Add a [fromList] generator
}
