/*
 * Copyright © 2018. Sir Wellington.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 * You may obtain a copy of the License at
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package tech.sirwellington.alchemy.generator

import org.slf4j.LoggerFactory
import tech.sirwellington.alchemy.annotations.access.NonInstantiable
import tech.sirwellington.alchemy.annotations.arguments.Required
import tech.sirwellington.alchemy.annotations.designs.patterns.StrategyPattern
import tech.sirwellington.alchemy.annotations.designs.patterns.StrategyPattern.Role.CONCRETE_BEHAVIOR
import tech.sirwellington.alchemy.generator.NumberGenerators.Companion.integers
import java.util.HashMap

/**
 * [Alchemy Generators][AlchemyGenerator] for Java Collections. These generators are useful in conjuction with
 * other [Generators][AlchemyGenerator], such as those in [NumberGenerators] or
 * [StringGenerators].
 *
 * @see StringGenerators
 *
 * @see NumberGenerators
 *
 * @see BinaryGenerators
 *
 *
 * @author SirWellington
 */
@NonInstantiable
@StrategyPattern(role = CONCRETE_BEHAVIOR)
class CollectionGenerators
@Throws(IllegalAccessException::class)
internal constructor()
{

    init
    {
        throw IllegalAccessException("cannot instantiate this class")
    }

    companion object
    {

        private val LOG = LoggerFactory.getLogger(CollectionGenerators::class.java)

        /**
         * Returns a list of Objects of varying size, using the supplied generator.
         *
         * @param <T>
         *
         * @param generator The generator that produces values
         *
         *
         * @return A list of random values, the length of which will vary.
         */
        @JvmStatic
        fun <T> listOf(@Required generator: AlchemyGenerator<T>): List<T>
        {
            val size = one(integers(5, 200))
            return listOf(generator, size)
        }

        /**
         * Returns a list of Objects using the supplied generator.
         *
         * @param <T>       The type to generate
         *
         * @param generator The generator that produces values
         *
         * @param size      The size of the list
         *
         *
         * @return A list of random values with length {code size}
         *
         *
         * @throws IllegalArgumentException if size is less than 0.
         */
        @JvmStatic
        fun <T> listOf(@Required generator: AlchemyGenerator<T>, size: Int): List<T>
        {
            checkThat(size >= 0, "Size must be at least 0")
            checkNotNull(generator, "generator is null")

            val list = List(size) {
                generator.get()
            }

            return list
        }

        /**
         * An [AlchemyGenerator] that returns values from a fixed list.
         *
         * @param <T>
         *
         * @param list
         *
         *
         * @return
         *
         * @throws IllegalArgumentException
         */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun <T> fromList(@Required list: List<T>): AlchemyGenerator<T>
        {
            checkNotNull(list, "list cannot be null")
            checkThat(list.isNotEmpty(), "list has no elements")

            return AlchemyGenerator {
                val index = one(integers(0, list.size))
                list[index]
            }

        }

        /**
         * Convenience method for [.mapOf].
         * Maps returned will vary in size.
         *
         * @param <K>
         * @param <V>
         * @param keyGenerator
         * @param valueGenerator
         *
         * @return
         *
         * @throws IllegalArgumentException
         */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun <K, V> mapOf(@Required keyGenerator: AlchemyGenerator<K>, @Required valueGenerator: AlchemyGenerator<V>): Map<K, V>
        {
            val size = one(integers(5, 100))
            return mapOf(keyGenerator, valueGenerator, size)
        }

        /**
         * Creates a Map using the Keys and Values generated by the provided Generators.
         *
         * @param <K>            The Type of the Keys
         * @param <V>            The Type of the Values
         *
         * @param keyGenerator   Generates keys for the Map
         * @param valueGenerator Generates values for the Map
         * @param size           The exact size of the created map
         *
         * @return Map generated from the parameters specified.Ï
         *
         * @throws IllegalArgumentException
         */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun <K, V> mapOf(@Required keyGenerator: AlchemyGenerator<K>, @Required valueGenerator: AlchemyGenerator<V>, size: Int): Map<K, V>
        {
            checkThat(size > 0, "size must be at least 1")
            checkNotNull(keyGenerator)
            checkNotNull(valueGenerator)

            val map = HashMap<K, V>(size)

            (0..size - 1).forEach {
                val key = keyGenerator.get()
                val value = valueGenerator.get()
                map.put(key, value)
            }

            return map
        }
    }

    //TODO:  Add a [fromList] generator
}
