package tech.yixiyun.framework.kuafu.controller.cookie;

import com.fasterxml.jackson.databind.JavaType;
import tech.yixiyun.framework.kuafu.controller.action.ActionContext;
import tech.yixiyun.framework.kuafu.kits.JSONKit;
import tech.yixiyun.framework.kuafu.kits.StringKit;

import javax.servlet.http.Cookie;
import java.util.Objects;

/**
 * Cookie工具类
 * @author Yixiyun
 * @version 1.0
 * @date 2021-06-01 10:08
 */
public class CookieKit {

    /**
     * 添加一个cookie
     * @param cookie
     */
    public static void addCookie(Cookie cookie) {
        ActionContext.getResponse().addCookie(cookie);
    }

    /**
     * 添加一个cookie，maxAge默认是-1，代表浏览器退出就删除
     * @param name
     * @param value
     */
    public static void addCookie(String name, String value) {
        addCookie(new Cookie(name, value));
    }

    /**
     * 添加一个cookie
     * @param name
     * @param value
     * @param maxAge cookie的存活时间，单位秒。负值代表不存储，浏览器关了就没了，正值代表存储，0代表立即删除
     */
    public static void addCookie(String name, String value, int maxAge) {
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        addCookie(cookie);
    }
    /**
     * 添加一个cookie
     * @param name
     * @param value
     * @param maxAge cookie的存活时间，负值代表不存储，浏览器关了就没了，正值代表存储，0代表立即删除
     * @param domain cookie生效的域名，以.开头的说明对父级域名都有效，例如.yixiyun.com ,表示对所有yixiyun.com的二三级域名都有效。
     *               不以.开头，则只对对应的域名有效
     */
    public static void addCookie(String name, String value, int maxAge, String domain) {
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        addCookie(cookie);
    }
    /**
     * 添加一个禁止js读取的cookie
     * @param name
     * @param value
     * @param maxAge cookie的存活时间，负值代表不存储，浏览器关了就没了，正值代表存储，0代表立即删除
     */
    public static void addCookieOnlyHttp(String name, String value, int maxAge) {
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        cookie.setHttpOnly(true);
        addCookie(cookie);
    }
    /**
     * 添加一个只允许https发送的cookie
     * @param name
     * @param value
     * @param maxAge cookie的存活时间，负值代表不存储，浏览器关了就没了，正值代表存储，0代表立即删除
     */
    public static void addCookieSecured(String name, String value, int maxAge) {
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        cookie.setSecure(true);
        addCookie(cookie);
    }
    /**
     * 添加一个最高安全等级的cookie，禁止js读取，并且只允许https发送到服务器
     * @param name
     * @param value
     * @param maxAge cookie的存活时间，负值代表不存储，浏览器关了就没了，正值代表存储，0代表立即删除
     */
    public static void addCookieHighSecured(String name, String value, int maxAge) {
        Cookie cookie = new Cookie(name, value);
        cookie.setMaxAge(maxAge);
        cookie.setSecure(true);
        cookie.setHttpOnly(true);
        addCookie(cookie);
    }





    /**
     * 移除某个cookie
     * @param name
     */
    public static void removeCookie(String name) {
        Cookie cookie = new Cookie(name, "");
        cookie.setMaxAge(0);
        addCookie(cookie);
    }

    /**
     * 取出Cookie值
     * @param name
     * @return
     */
    public static String getCookie(String name) {
        Cookie[] cookies = ActionContext.getRequest().getCookies();
        for (Cookie cookie : cookies) {
            if (Objects.equals(cookie.getName(), name)) {
                return cookie.getValue();
            }
        }

        return  null;


    }

    /**
     * 如果cookie的值是一个json字符串，根据传入的javatype，进行转化，返回对应类型的对象
     * @param name
     * @param type
     * @param <T>
     * @return
     */
    public static <T> T getCookieToObject(String name, JavaType type) {
        String cookie = getCookie(name);
        if (StringKit.isNotBlank(cookie)) {
            return (T) JSONKit.toObject(cookie, type);
        }

        return null;
    }





}
