package tech.yixiyun.framework.kuafu.controller.request.param;


import tech.yixiyun.framework.kuafu.controller.annotation.Cookie;
import tech.yixiyun.framework.kuafu.controller.annotation.Session;
import tech.yixiyun.framework.kuafu.controller.annotation.UrlParamIndex;
import tech.yixiyun.framework.kuafu.kits.StringKit;

import java.lang.reflect.*;
import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 请求的方法参数包装类
 */
public class ParamDefinition {

    /**
     * 记录类型的特征，防止循环依赖解析时，导致死循环
     */
    private static final ConcurrentHashMap<Type, ParamFeature[]> TYPE_FEATURES = new ConcurrentHashMap<>();


    /**
     * 参数名
     */
    private String name;
    /**
     * 参数是否来自url参数，类似 /user/1-true 这种方式的传入的参数。
     * 默认是-1，代表不是url参数，>-1时，就代表是url参数
     */
    private int urlParamIndex = -1;

    /**
     * 参数是否来自session绑定，如果为null，说明不是，
     * 不是null说明是，并且值就是用于绑定的name
     */
    private String sessionAttrName;


    /**
     * 参数是否来自cookie绑定，如果为null，说明不是，
     * 不是null说明是，并且值就是用于绑定的name
     */
    private String cookieAttrName;



    /**
     * 方法参数
     */
    private Parameter parameter;


    private ParamFeature paramFeature;



    public ParamDefinition(Parameter param) {

        this.parameter = param;

        resolveParam();
    }




    /**
     * 解析这个参数
     */
    private void resolveParam() {
        this.name = this.parameter.getName();
        //参数值是否来自url
        UrlParamIndex index;
        Session session;
        Cookie cookie;
        if ((index = this.parameter.getDeclaredAnnotation(UrlParamIndex.class)) != null) {
            this.urlParamIndex = index.value();
        } else if ((session = this.parameter.getDeclaredAnnotation(Session.class)) != null) {
            this.sessionAttrName = StringKit.isBlank(session.value()) ? name : session.value();
        } else if((cookie = this.parameter.getDeclaredAnnotation(Cookie.class)) != null) {
            this.cookieAttrName = StringKit.isBlank(cookie.value())? name : cookie.value();
        }


        this.paramFeature = analyseParamFeatrue(this.parameter.getParameterizedType());

    }



    /**
     * 解析参数特征
     * @param type
     */
    private ParamFeature analyseParamFeatrue(Type type) {
        ParamFeature feature = new ParamFeature();

        Class clazz = null;

        if (type instanceof Class) {

            clazz = (Class) type;
            feature.setParamType(ParamType.analyseParamType(clazz));

        } else if (type instanceof ParameterizedType) {

            clazz = (Class)((ParameterizedType) type).getRawType();
            feature.setParamType(ParamType.analyseParamType(clazz));

        } else if(type instanceof GenericArrayType) {

            feature.setParamType(ParamType.ARRAY);

        } else {

            throw new RuntimeException("不支持进行参数转化的类型：" + type);

        }

        feature.setClazz(clazz);


        if (feature.getParamType() != ParamType.COMMON) {
            feature.setSubParamFeatures(analyzeSubFeatures(feature.getParamType(), type));
        }
        return feature;
    }


    /**
     * 解析子函数并生成特征
     * @param paramType
     * @param type
     * @return
     */
    private ParamFeature[] analyzeSubFeatures(ParamType paramType, Type type) {

        switch (paramType) {
            case POJO:

                return analyzePojoFeatures(type);

            case LIST:

                return analyzeListElementFeatures(type);

            case MAP:

                return analyzeMapEntryFeatures(type);

            case ARRAY:

                return analyzeArrayComponentFeatures(type);
        }
        return null;
    }

    /**
     * 解析POJO成员变量特征
     * @param pojoType
     * @return
     */
    private ParamFeature[] analyzePojoFeatures(Type pojoType) {

        if (pojoType instanceof Class == false) {
            throw new RuntimeException("无法识别解析类型：" + pojoType);
        }
        Field[] fields = (Field[]) Arrays.stream(((Class) pojoType).getDeclaredFields()).filter(item -> {
            int modifiers = item.getModifiers();
            return Modifier.isStatic(modifiers) == false && Modifier.isFinal(modifiers) == false;
        }).toArray(Field[]::new);


        ParamFeature[] features = new ParamFeature[fields.length];
        ParamFeature[] exist = TYPE_FEATURES.putIfAbsent(pojoType, features);
        //map中原来解析过，就直接用
        if (exist != null) {
            return exist;
        }

        for (int i = 0; i < fields.length; i++) {
            Field field = fields[i];
            field.setAccessible(true);
            ParamFeature feature = analyseParamFeatrue(field.getGenericType());
            feature.setField(field);
            features[i] = feature;
            
        }
        return features;


    }



    /**
     * 解析Map记录的特征
     * @param mapType
     * @return
     */
    private ParamFeature[] analyzeMapEntryFeatures(Type mapType) {
        if (mapType instanceof ParameterizedType == false) {
            throw new RuntimeException("必须指定具体的泛型类型才可进行参数转化：" + mapType);
        }
        Type[] types = ((ParameterizedType)mapType).getActualTypeArguments();
        return new ParamFeature[]{
                analyseParamFeatrue(types[0]),
                analyseParamFeatrue(types[1])
        };
    }

    /**
     * 解析List的元素特征
     * @param listType
     * @return
     */
    private ParamFeature[] analyzeListElementFeatures(Type listType) {
        if (listType instanceof ParameterizedType == false) {
            throw new RuntimeException("必须指定具体的泛型类型才可进行参数转化："  + listType);
        }
        return new ParamFeature[]{
                analyseParamFeatrue(((ParameterizedType)listType).getActualTypeArguments()[0])
        };
    }

    /**
     * 解析数组的子元素特征
     * @param arrayType
     * @return
     */
    private ParamFeature[] analyzeArrayComponentFeatures(Type arrayType) {
        Type componentType = null;
        if (arrayType instanceof Class) {
            componentType = ((Class)arrayType).getComponentType();
        } else if(arrayType instanceof GenericArrayType) {
            componentType = ((GenericArrayType)arrayType).getGenericComponentType();
        }
        return new ParamFeature[]{analyseParamFeatrue(componentType)};
    }

    public String getName() {
        return name;
    }

    public int getUrlParamIndex() {
        return urlParamIndex;
    }

    public Parameter getParameter() {
        return parameter;
    }

    public ParamFeature getParamFeature() {
        return this.paramFeature;
    }

    public String getSessionAttrName() {
        return sessionAttrName;
    }

    public String getCookieAttrName() {
        return cookieAttrName;
    }

    /**
     * 参数特征
     */
    public static class ParamFeature {
        private ParamType type; //类型
        private Class clazz; //参数类
        private Field field; //是哪个字段，只有当父级paramType是 POJO时，才会用到

        /**
         * 子参数类型特征，<br/>
         * 如果paramType = Array，这里存放的就是数组元素的类型特征 <br/>
         * 如果paramType = List，这里存放的就是集合元素的类型特征 <br/>
         * 如果paramType = Map, 这里存放的就是Key 和 Value的类型特征 <br/>
         * 如果paramType = POJO, 这里存放的就是成员变量的类型特征
         */
        private ParamFeature[] subParamFeatures;


        public ParamType getParamType() {
            return type;
        }

        public void setParamType(ParamType paramType) {
            this.type = paramType;
        }

        public Class getClazz() {
            return clazz;
        }

        public void setClazz(Class clazz) {
            this.clazz = clazz;
        }

        public Field getField() {
            return field;
        }

        public void setField(Field field) {
            this.field = field;
        }

        public ParamFeature[] getSubParamFeatures() {
            return subParamFeatures;
        }

        public void setSubParamFeatures(ParamFeature[] subParamFeatures) {
            this.subParamFeatures = subParamFeatures;
        }
    }

}
