package tech.yixiyun.framework.kuafu.db.datasource;


import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.pool.DruidDataSourceFactory;
import com.fasterxml.jackson.databind.JsonNode;
import tech.yixiyun.framework.kuafu.config.AppConfig;
import tech.yixiyun.framework.kuafu.db.DbException;
import tech.yixiyun.framework.kuafu.kits.JSONKit;
import tech.yixiyun.framework.kuafu.kits.StringKit;

/**
 * 阿里Druid数据源包装类。可以直接通过配置项的key完成初始化配置。<br/>
 * <b>name</b>：为数据源起的名字，如果系统要使用多数据源，就通过name来调用 <br/>
 * <b>isMain</b>：是否是主数据源，一个系统只能有一个主数据源。当执行Sql语句时，如果未指定数据源，就默认使用主数据源。<br/>
 * <b>dataSource</b>：数据源实例 <br/>
 *
 */
public class DruidDataSourceDefinition extends DataSourceDefinition{


    @Override
    public void close() {
        DruidDataSource ds = (DruidDataSource) this.getDataSource();
        if (ds != null) {
            ds.close();
        }
    }

    public DruidDataSourceDefinition(String name, DbType dbType, DruidDataSource dataSource, boolean isMain) {
        super(name, dbType, dataSource, isMain);
    }

    public DruidDataSourceDefinition(String configKey) {
        this(AppConfig.getAsTreeNode(configKey));
    }


    public DruidDataSourceDefinition(JsonNode config) {
        String name = config.get("name").asText();
        if (StringKit.isBlank(name) ) {
            throw new DbException("数据源未配置name（为数据源起的名字），注册失败");
        }

        String url = config.get("url").asText();
        if (StringKit.isBlank(url) ) {
            throw new DbException("数据源未配置url（数据库连接），注册失败");
        }
        String username = config.get("username").asText();
        if (StringKit.isBlank(username) ) {
            throw new DbException("数据源未配置username（用户名），注册失败");
        }
        String password = config.get("password").asText();
        if (StringKit.isBlank(password) ) {
            throw new DbException("数据源未配置password（密码），注册失败");
        }

        this.setName(name);
        try {
            DruidDataSource dataSource = (DruidDataSource) DruidDataSourceFactory.createDataSource(JSONKit.toStringMap(config.toString()));
            this.setDataSource(dataSource);
            dataSource.init();
            //更新数据库类型，如果配置了dbType，就按配置的来
            JsonNode node = config.get("dbType");
            if (node != null &&  StringKit.isNotBlank(node.asText())) {
                this.setDbType(DbType.getByName(node.asText()));
            } else { //否则自动识别
                //只有init之后，才能获取到值
                this.setDbType(DbType.getByName(dataSource.getDbType()));
            }



        } catch (Exception e) {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException(e);
        }
    }

    public DruidDataSource getDruidDataSource() {
        return (DruidDataSource) this.dataSource;
    }



}
