package com.wu.framework.inner.lazy.database.expand.database.persistence.factory;

import com.wu.framework.inner.database.dynamic.LazyDynamicAdapter;
import com.wu.framework.inner.database.dynamic.factory.LazyDynamicAdapterFactory;
import com.wu.framework.inner.lazy.config.LazyOperationConfig;
import com.wu.framework.inner.lazy.config.prop.LazyDataSourceProperties;
import com.wu.framework.inner.lazy.database.expand.database.persistence.LazyOperation;
import com.wu.framework.inner.lazy.database.expand.database.persistence.method.*;
import com.wu.framework.inner.lazy.database.expand.database.persistence.proxy.LazyOperationProxy;
import org.springframework.context.ApplicationContext;

import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.List;

/**
 * LazyOperationProxy 代理工厂
 */
public final class LazyOperationProxyFactory {


    /**
     * 创建默认的操作代理对象
     *
     * @param sourceProperties 数据库链接配置
     * @return
     */
    public static LazyOperationProxy createDefaultLazyOperationProxy(LazyDataSourceProperties sourceProperties) {
        return createDefaultLazyOperationProxy(sourceProperties, null);
    }

    /**
     * 创建默认的操作代理对象
     *
     * @param sourceProperties    数据库链接配置
     * @param lazyOperationConfig 操作配置
     * @return
     */
    public static LazyOperationProxy createDefaultLazyOperationProxy(LazyDataSourceProperties sourceProperties,
                                                                     LazyOperationConfig lazyOperationConfig) {
        List<LazyOperationMethod> lazyOperationMethods = Arrays.asList(
                new LazyOperationMethodCreateTable(),
                new LazyOperationMethodExecute(),
                new LazyOperationMethodExecuteOne(),
                new LazyOperationMethodExecuteSQL(lazyOperationConfig),
                new LazyOperationMethodExecuteSQLForBean(lazyOperationConfig),
                new LazyOperationMethodInsert(lazyOperationConfig),
                new LazyOperationMethodPage(lazyOperationConfig),
                new LazyOperationMethodPerfect(lazyOperationConfig),
                new LazyOperationMethodSmartUpsert(lazyOperationConfig),
                new LazyOperationMethodUpdateTable(),
                new LazyOperationMethodUpsert(lazyOperationConfig),
                new LazyOperationMethodUpsertRemoveNull(lazyOperationConfig),
                new LazyOperationMethodScriptRunner(lazyOperationConfig)

        );
        return createLazyOperationProxy(null, sourceProperties, lazyOperationMethods);
    }


    /**
     * 创建默认的操作代理对象
     *
     * @param sourceProperties     数据库链接配置
     * @param lazyOperationMethods 操作的方法
     * @return
     */
    public static LazyOperationProxy createLazyOperationProxy(LazyDataSourceProperties sourceProperties,
                                                              List<LazyOperationMethod> lazyOperationMethods) {
        return createLazyOperationProxy(null, sourceProperties, lazyOperationMethods);
    }

    /**
     * 创建默认的操作代理对象
     *
     * @param sourceProperties     数据库链接配置
     * @param lazyOperationMethods 操作的方法
     * @param applicationContext   上下文
     * @return
     */
    public static LazyOperationProxy createLazyOperationProxy(ApplicationContext applicationContext,
                                                              LazyDataSourceProperties sourceProperties,
                                                              List<LazyOperationMethod> lazyOperationMethods
    ) {
        LazyDynamicAdapter lazyDynamicAdapter = LazyDynamicAdapterFactory.createLazyDynamicAdapter(applicationContext, sourceProperties);
        LazyOperationProxy lazyOperationProxy = new LazyOperationProxy(lazyOperationMethods, lazyDynamicAdapter);
        try {
            lazyOperationProxy.afterPropertiesSet();
        } catch (Exception exception) {
            exception.printStackTrace();
        }
        return lazyOperationProxy;
    }


    /**
     * 创建对象操作
     *
     * @param lazyDataSourceProperties 数据库链接配置
     * @return
     */
    public static LazyOperation createLazyOperation(LazyDataSourceProperties lazyDataSourceProperties) {
        LazyOperationProxy defaultLazyOperationProxy = createDefaultLazyOperationProxy(lazyDataSourceProperties);
        return createLazyOperation(defaultLazyOperationProxy);
    }

    /**
     * 创建对象操作
     *
     * @param lazyOperationProxy 操作对象代理对象
     * @return
     */
    public static LazyOperation createLazyOperation(LazyOperationProxy lazyOperationProxy) {
        return (LazyOperation) Proxy.newProxyInstance(LazyOperation.class.getClassLoader(), new Class[]{LazyOperation.class}, lazyOperationProxy);
    }

    /**
     * 创建 LazyOperation 对象
     *
     * @param url      数据库地址
     * @param username 用户名称
     * @param password 用户密码
     * @return 自动填充数据对象
     * @return
     */
    public static LazyOperation createLazyOperation(String url, String username, String password) {
        LazyDataSourceProperties lazyDataSourceProperties = new LazyDataSourceProperties();
        lazyDataSourceProperties.setUrl(url);
        lazyDataSourceProperties.setUsername(username);
        lazyDataSourceProperties.setPassword(password);
        return createLazyOperation(lazyDataSourceProperties);
    }

    /**
     * 创建 LazyOperation 对象
     *
     * @param host     数据库IP
     * @param port     数据库端口
     * @param username 用户名称
     * @param password 用户密码
     * @return 自动填充数据对象
     * @return
     */
    public static LazyOperation createLazyOperation(String host, int port, String schema, String username, String password) {
        String urlFormat = "jdbc:mysql://%s:%s/%s?useUnicode=true&characterEncoding=utf-8&useSSL=true&allowMultiQueries=true&serverTimezone=Asia/Shanghai";
        String url = String.format(urlFormat, host, port, schema);

        return createLazyOperation(url, username, password);
    }


}
