package com.wu.framework.inner.lazy.database.expand.database.persistence.proxy;

import com.wu.framework.inner.database.dynamic.LazyDynamicAdapter;
import com.wu.framework.inner.database.dynamic.toolkit.DynamicLazyDSContextHolder;
import com.wu.framework.inner.layer.stereotype.proxy.ProxyStrategicApproach;
import com.wu.framework.inner.lazy.database.expand.database.persistence.method.LazyOperationMethod;
import com.wu.framework.inner.lazy.database.smart.database.persistence.PerfectLazyOperation;
import com.wu.framework.inner.lazy.persistence.conf.LazyDynamicEndpoint;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Import;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.util.ObjectUtils;

import javax.sql.DataSource;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author : Jia wei Wu
 * @version 1.0
 * describe : 自定义接口实现方法执行代理类
 * @date : 2020/6/25 下午11:19
 */
@Slf4j
@ConditionalOnBean(value = DataSource.class)
@Import(PerfectLazyOperation.class)
public class LazyOperationProxy implements InvocationHandler, InitializingBean {

    private final static Map<Class<? extends LazyOperationMethod>, LazyOperationMethod> LAZY_OPERATION_METHOD_MAP = new HashMap<>();

    private final List<LazyOperationMethod> lazyOperationMethods;

    private final LazyDynamicAdapter lazyDynamicAdapter;


    public LazyOperationProxy(List<LazyOperationMethod> lazyOperationMethods, LazyDynamicAdapter lazyDynamicAdapter) {
        this.lazyOperationMethods = lazyOperationMethods;
        this.lazyDynamicAdapter = lazyDynamicAdapter;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        ProxyStrategicApproach mergedAnnotation = AnnotatedElementUtils.findMergedAnnotation(method, ProxyStrategicApproach.class);
        if (null != mergedAnnotation) {
            Connection connection;
            boolean isConnectionTransactional;

            boolean autoCommit;
            LazyOperationMethod lazyOperationMethod = LAZY_OPERATION_METHOD_MAP.get(mergedAnnotation.proxyClass());
            if (null == lazyOperationMethod) {
                log.error("无法找到对应class ：【{}】的代理实现", mergedAnnotation.proxyClass());
            }
            final DataSource dataSource = lazyDynamicAdapter.determineDataSource();
            // 切换数据库
            connection = DataSourceUtils.getConnection(dataSource);
            switchSchema(connection);
            isConnectionTransactional = DataSourceUtils.isConnectionTransactional(connection, dataSource);
            autoCommit = connection.getAutoCommit();
            try {
                if (connection.isClosed()) {
                    System.out.println("this.connection  关闭了");
                }
                // 判断不是事物重新获取链接
                final Object execute = lazyOperationMethod.execute(connection, args);
                // 是否提交
                if (connection != null && !isConnectionTransactional && !autoCommit) {
                    log.debug("Committing JDBC Connection [" + connection + "]");
                    connection.commit();
                }
                return execute;
            } catch (Exception exception) {
                // 事物回滚
                if (connection != null && !isConnectionTransactional && !autoCommit) {
                    log.debug("Rolling back JDBC Connection [" + connection + "]");
                    connection.rollback();
                }
                exception.printStackTrace();
                throw exception;
            } finally {
                // 释放链接
                DataSourceUtils.releaseConnection(connection, dataSource);
            }
        } else {
            if (method.getParameterCount() == 0) {
                return method.invoke(this, args);
            } else {
                return method.invoke(null, args);
            }
        }
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        lazyOperationMethods.forEach(lazyOperationMethod -> LAZY_OPERATION_METHOD_MAP.put(lazyOperationMethod.getClass(), lazyOperationMethod));
    }

    /**
     * 确定数据源
     *
     * @return
     * @author Jia wei Wu
     * @date 2022/1/1 5:02 下午
     **/
    public DataSource determineConnection() throws SQLException {
        return lazyDynamicAdapter.determineDataSource();
    }

    /**
     * 切换数据库
     *
     * @param connection 链接对象
     */
    public Connection switchSchema(Connection connection) {
        LazyDynamicEndpoint ds = DynamicLazyDSContextHolder.peek();
        if (null != ds && !ObjectUtils.isEmpty(ds.getSchema())) {
            String schema = ds.getSchema();
            try {
                connection.setSchema(schema);
            } catch (SQLException e) {
                e.printStackTrace();
                log.error("切换数据库失败:【{}】", e.getMessage());
            }
        }
        return connection;
    }
}
