package com.wu.framework.inner.lazy.database.expand.database.persistence.proxy;

import com.wu.framework.inner.layer.stereotype.proxy.ProxyStrategicApproach;
import com.wu.framework.inner.lazy.database.dynamic.LazyDynamicAdapter;
import com.wu.framework.inner.lazy.database.expand.database.persistence.method.LazyOperationMethod;
import com.wu.framework.inner.lazy.database.smart.database.persistence.PerfectLazyOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.annotation.Import;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.jdbc.datasource.DataSourceUtils;

import javax.sql.DataSource;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author : Jia wei Wu
 * @version 1.0
 * describe : 自定义接口实现方法执行代理类
 * @date : 2020/6/25 下午11:19
 */
@Slf4j
@ConditionalOnBean(value = DataSource.class)
@Import(PerfectLazyOperation.class)
public class LazyOperationProxy implements InvocationHandler, InitializingBean {

    private final static Map<Class<? extends LazyOperationMethod>, LazyOperationMethod> LAZY_OPERATION_METHOD_MAP = new HashMap<>();

    private final List<LazyOperationMethod> lazyOperationMethods;

    private final LazyDynamicAdapter lazyDynamicAdapter;


    private boolean isConnectionTransactional;

    private boolean autoCommit;

    private Connection connection;


    public LazyOperationProxy(List<LazyOperationMethod> lazyOperationMethods, LazyDynamicAdapter lazyDynamicAdapter) {
        this.lazyOperationMethods = lazyOperationMethods;
        this.lazyDynamicAdapter = lazyDynamicAdapter;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        ProxyStrategicApproach mergedAnnotation = AnnotatedElementUtils.findMergedAnnotation(method, ProxyStrategicApproach.class);
        if (null != mergedAnnotation) {
            LazyOperationMethod lazyOperationMethod = LAZY_OPERATION_METHOD_MAP.get(mergedAnnotation.proxyClass());
            // TODO 动态数据源切换
            final DataSource dataSource = lazyDynamicAdapter.determineDataSource();
            this.connection = DataSourceUtils.getConnection(dataSource);
            this.isConnectionTransactional = DataSourceUtils.isConnectionTransactional(this.connection, dataSource);
            this.autoCommit = connection.getAutoCommit();
            try {
                // 判断不是事物重新获取链接
                final Object execute = lazyOperationMethod.execute(this.connection, args);
                // 是否提交
                if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
                    log.debug("Committing JDBC Connection [" + this.connection + "]");
                    this.connection.commit();
                }
                return execute;
            } catch (Exception exception) {
                // 实物回滚
                if (this.connection != null && !this.isConnectionTransactional && !this.autoCommit) {
                    log.debug("Rolling back JDBC Connection [" + this.connection + "]");
                    this.connection.rollback();
                }
                exception.printStackTrace();
                throw exception;
            } finally {
                // 释放链接
                DataSourceUtils.releaseConnection(connection, dataSource);
            }
        } else {
            if (method.getParameterCount() == 0) {
                return method.invoke(this, args);
            } else {
                return method.invoke(null, args);
            }
        }
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        lazyOperationMethods.forEach(lazyOperationMethod -> LAZY_OPERATION_METHOD_MAP.put(lazyOperationMethod.getClass(), lazyOperationMethod));
    }

    /**
     * 确定数据源
     *
     * @return
     * @author Jia wei Wu
     * @date 2022/1/1 5:02 下午
     **/
    public DataSource determineConnection() throws SQLException {
        return lazyDynamicAdapter.determineDataSource();
    }

}
