/*
 * MIT License
 *
 * Copyright (c) 2018 MODL
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package uk.num.json_modl.converter;

import java.util.function.Function;
import java.util.stream.Collectors;

class UnicodeEscaper {

    /**
     * Unicode escapes for non-ASCII values.
     */
    static Function<String, String> escape = (s) -> (s == null) ? null : s.codePoints()
            .mapToObj(UnicodeEscaper::escape)
            .collect(Collectors.joining());

    /**
     * Unicode escapes for non-ASCII characters.
     *
     * @param c the codepoint to be checked
     * @return a String with escaped unicode sequences if necessary.
     */
    private static String escape(final int c) {
        if (c >= 32 && c <= 127) {
            return String.valueOf(Character.toChars(c));
        } else if (c >= 0x100000 && c <= 0x10ffff) {
            return String.format("~u%06x", c);
        } else if (c == '\b') {
            return "\\b";
        } else if (c == '\f') {
            return "\\f";
        } else if (c == '\n') {
            return "\\n";
        } else if (c == '\r') {
            return "\\r";
        } else if (c == '\t') {
            return "\\t";
        } else if (c >= 0x10000 && c <= 0xfffff) {
            return String.format("~u%05x", c);
        } else {
            return String.format("~u%04x", c);
        }
    }

}
